% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_reindex_tips.R
\name{tree_reindex_tips}
\alias{tree_reindex_tips}
\title{Reindex tree tips in order of appearance in edges}
\usage{
tree_reindex_tips(tree)
}
\arguments{
\item{tree}{A \code{phylo} object from the \code{ape} package (see \code{\link[ape:read.tree]{ape::read.tree()}}).
Works with standard \code{phylo} objects, and also with our extended trees (in that additive edges \code{tree$edge.length.add} are recalculated after reordering if they were present).}
}
\value{
The modified \code{tree} (\code{phylo} object) with reordered edges and tips.
}
\description{
The \code{phylo} object from the \code{ape} package (see \code{\link[ape:read.tree]{ape::read.tree()}}) permits mismatches between the order of tips as present in the tip labels vector (\code{tree$tip.label}) and in the edge matrix (\code{tree$edge}), which can cause mismatches in plots and other settings.
This function reindexes edges and tips so that they agree in tip order.
}
\details{
Order mismatches between tip labels vector and edge matrix can lead to disagreeing order downstream, in variables that order tips as in the labels vector (such as the coancestry matrices output by our \code{\link[=coanc_tree]{coanc_tree()}}) and tree plots (whose order is guided by the edge matrix, particularly when the edge matrix is ordered by clade as in \code{\link[ape:reorder.phylo]{ape::reorder.phylo()}}).

This function first reorders the edges of the input tree using \code{\link[ape:reorder.phylo]{ape::reorder.phylo()}} with default option \code{order = 'cladewise'}, which will list edges and tip nodes in plotting order.
Then tips are indexed so that the first tip to appear has index 1 in the edge matrix (and consequently appears first in the tip labels vector), the second has index 2, and so on.
Thus, the output tree has both edges and tips reordered, to have a consistent tip order and best experience visualizing trees and their coancestry matrices.
}
\examples{
# create a random tree
library(ape)
k <- 5
tree <- rtree( k )

# trees constructed this way are already ordered as desired, so this function doesn't change it:
tree2 <- tree_reindex_tips( tree )
# tree2 equals tree!

# let's scramble the edges on purpose
# to create an example where reindexing is needed

tree_rand <- tree
# new order of edges
indexes <- sample( Nedge( tree_rand ) )
# reorder all edge values
tree_rand$edge <- tree_rand$edge[ indexes, ]
tree_rand$edge.length <- tree_rand$edge.length[ indexes ]
# now let's reorder edges slightly so tree is more reasonable-looking
# (otherwise plot looks tangled)
tree_rand <- reorder( tree_rand, order = 'postorder' )
# you can now see that, unless permutation got lucky,
# the order of the tip labels in the vector and on the plot disagree:
tree_rand$tip.label
plot( tree_rand )

# now reorder tips to match plotting order (as defined in the `edge` matrix)
tree_rand <- tree_reindex_tips( tree_rand )
# now tip labels vector and plot should agree in order:
# (plot was not changed)
tree_rand$tip.label
plot( tree_rand )

}
\seealso{
\code{\link[=tree_reorder]{tree_reorder()}} for reordering tree structure so that tips appear in a particular desired order.
}
