% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nhclu_affprop.R
\name{nhclu_affprop}
\alias{nhclu_affprop}
\title{Non-hierarchical clustering: Affinity Propagation}
\usage{
nhclu_affprop(
  similarity,
  index = names(similarity)[3],
  seed = NULL,
  p = NA,
  q = NA,
  maxits = 1000,
  convits = 100,
  lam = 0.9,
  details = FALSE,
  nonoise = FALSE,
  K = NULL,
  prc = NULL,
  bimaxit = NULL,
  exact = NULL,
  algorithm_in_output = TRUE
)
}
\arguments{
\item{similarity}{The output object from \code{\link[=similarity]{similarity()}} or
\code{\link[=dissimilarity_to_similarity]{dissimilarity_to_similarity()}}, or a \code{dist} object. If a \code{data.frame} is
used, the first two columns should represent pairs of sites (or any pair of
nodes), and the subsequent column(s) should contain the similarity indices.}

\item{index}{The name or number of the similarity column to use. By default,
the third column name of \code{similarity} is used.}

\item{seed}{The seed for the random number generator used when
\code{nonoise = FALSE}.}

\item{p}{Input preference, which can be a vector specifying individual
preferences for each data point. If scalar, the same value is used for all
data points. If \code{NA}, exemplar preferences are initialized based on the
distribution of non-Inf values in the similarity matrix, controlled by \code{q}.}

\item{q}{If \code{p = NA}, exemplar preferences are initialized according to the
distribution of non-Inf values in the similarity matrix. By default, the
median is used. A value between 0 and 1 specifies the sample quantile,
where \code{q = 0.5} results in the median.}

\item{maxits}{The maximum number of iterations to execute.}

\item{convits}{The algorithm terminates if the exemplars do not change for
\code{convits} iterations.}

\item{lam}{The damping factor, a value in the range [0.5, 1). Higher values
correspond to heavier damping, which may help prevent oscillations.}

\item{details}{If \code{TRUE}, detailed information about the algorithm's progress
is stored in the output object.}

\item{nonoise}{If \code{TRUE}, disables the addition of a small amount of noise to
the similarity object, which prevents degenerate cases.}

\item{K}{The desired number of clusters. If not \code{NULL}, the function
\link[apcluster:apclusterK-methods]{apclusterK} is called.}

\item{prc}{A parameter needed when \code{K} is not \code{NULL}. The algorithm stops if
the number of clusters deviates by less than \code{prc} percent from the desired
value \code{K}. Set to 0 to enforce exactly \code{K} clusters.}

\item{bimaxit}{A parameter needed when \code{K} is not \code{NULL}. Specifies the
maximum number of bisection steps to perform. No warning is issued if the
number of clusters remains outside the desired range.}

\item{exact}{A flag indicating whether to compute the initial preference
range exactly.}

\item{algorithm_in_output}{A \code{boolean} indicating whether to include the
original output of \link[apcluster:apcluster-methods]{apcluster} in the result. Defaults
to \code{TRUE}.}
}
\value{
A \code{list} of class \code{bioregion.clusters} with five slots:
\enumerate{
\item{\strong{name}: A \code{character} string containing the name of the algorithm.}
\item{\strong{args}: A \code{list} of input arguments as provided by the user.}
\item{\strong{inputs}: A \code{list} describing the characteristics of the clustering
process.}
\item{\strong{algorithm}: A \code{list} of objects associated with the clustering
procedure, such as original cluster objects
(if \code{algorithm_in_output = TRUE}).}
\item{\strong{clusters}: A \code{data.frame} containing the clustering results.}}

If \code{algorithm_in_output = TRUE}, the \code{algorithm} slot includes the output of
\link[apcluster:apcluster-methods]{apcluster}.
}
\description{
This function performs non-hierarchical clustering using the Affinity
Propagation algorithm.
}
\details{
This function is based on the
\href{https://cran.r-project.org/package=apcluster}{apcluster}
package (\link[apcluster:apcluster-methods]{apcluster}).
}
\examples{
comat_1 <- matrix(sample(0:1000, size = 10*12, replace = TRUE,
prob = 1/1:1001), 10, 12)
rownames(comat_1) <- paste0("Site", 1:10)
colnames(comat_1) <- paste0("Species", 1:12)
comat_1 <- cbind(comat_1,
                 matrix(0, 10, 8,
                        dimnames = list(paste0("Site", 1:10),
                                        paste0("Species", 13:20))))
                                        
comat_2 <- matrix(sample(0:1000, 
                         size = 10*12, 
                         replace = TRUE, 
                         prob = 1/1:1001), 
                  10, 12)
rownames(comat_2) <- paste0("Site", 11:20)
colnames(comat_2) <- paste0("Species", 9:20)
comat_2 <- cbind(matrix(0, 10, 8, 
                        dimnames = list(paste0("Site", 11:20),
                                        paste0("Species", 1:8))),
                 comat_2)
                 
comat <- rbind(comat_1, comat_2)

dissim <- dissimilarity(comat, metric = "Simpson")
sim <- dissimilarity_to_similarity(dissim)

clust1 <- nhclu_affprop(sim)

clust2 <- nhclu_affprop(sim, q = 1)

# Fixed number of clusters 
clust3 <- nhclu_affprop(sim, K = 2, prc = 10, bimaxit = 20, exact = FALSE)

}
\references{
Frey B & Dueck D (2007) Clustering by Passing Messages Between Data Points.
\emph{Science} 315, 972-976.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a4_2_non_hierarchical_clustering.html}.

Associated functions:
\link{nhclu_clara} \link{nhclu_clarans} \link{nhclu_dbscan} \link{nhclu_kmeans} \link{nhclu_affprop}
}
\author{
Pierre Denelle (\email{pierre.denelle@gmail.com}) \cr
Boris Leroy (\email{leroy.boris@gmail.com}) \cr
Maxime Lenormand (\email{maxime.lenormand@inrae.fr})
}
