% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compare_bioregionalizations.R
\name{compare_bioregionalizations}
\alias{compare_bioregionalizations}
\title{Compare cluster memberships among multiple bioregionalizations}
\usage{
compare_bioregionalizations(
  bioregionalizations,
  indices = c("rand", "jaccard"),
  cor_frequency = FALSE,
  store_pairwise_membership = TRUE,
  store_confusion_matrix = TRUE
)
}
\arguments{
\item{bioregionalizations}{A \code{data.frame} object where each row corresponds to
a site, and each column to a bioregionalization.}

\item{indices}{\code{NULL} or \code{character}. Indices to compute for the pairwise
comparison of bioregionalizations. Currently available metrics are \code{"rand"}
and \code{"jaccard"}.}

\item{cor_frequency}{A \code{boolean}. If \code{TRUE}, computes the correlation
between each bioregionalization and the total frequency of co-membership of
items across all bioregionalizations. This is useful for identifying which
bioregionalization(s) is(are) most representative of all computed
bioregionalizations.}

\item{store_pairwise_membership}{A \code{boolean}. If \code{TRUE}, stores the pairwise
membership of items in the output object.}

\item{store_confusion_matrix}{A \code{boolean}. If \code{TRUE}, stores the confusion
matrices of pairwise bioregionalization comparisons in the output object.}
}
\value{
A \code{list} containing 4 to 7 elements:

\enumerate{
\item{\strong{args}: A \code{list} of user-provided arguments.}
\item{\strong{inputs}: A \code{list} containing information on the input
bioregionalizations, such as the number of items clustered.}
\item{\strong{pairwise_membership} (optional): If \code{store_pairwise_membership = TRUE},
a \verb{boolean matrix} where \code{TRUE} indicates two items are in the same cluster,
and \code{FALSE} indicates they are not.}
\item{\strong{freq_item_pw_membership}: A \verb{numeric vector} containing the number of
times each item pair is clustered together, corresponding to the sum of rows
in \code{pairwise_membership}.}
\item{\strong{bioregionalization_freq_cor} (optional): If \code{cor_frequency = TRUE},
a \verb{numeric vector} of correlations between individual bioregionalizations
and the total frequency of pairwise membership.}
\item{\strong{confusion_matrix} (optional): If \code{store_confusion_matrix = TRUE},
a \code{list} of confusion matrices for each pair of bioregionalizations.}
\item{\strong{bioregionalization_comparison}: A \code{data.frame} containing comparison
results, where the first column indicates the bioregionalizations compared,
and the remaining columns contain the requested \code{indices}.}}
}
\description{
This function computes pairwise comparisons for several
bioregionalizations, usually outputs from \code{netclu_}, \code{hclu_}, or \code{nhclu_}
functions. It also provides the confusion matrix from pairwise comparisons,
enabling the user to compute additional comparison metrics.
}
\details{
This function operates in two main steps:
\enumerate{
\item Within each bioregionalization, the function compares all pairs of items
and documents whether they are clustered together (\code{TRUE}) or separately
(\code{FALSE}). For example, if site 1 and site 2 are clustered in the same
cluster in bioregionalization 1, their pairwise membership \code{site1_site2}
will be \code{TRUE}. This output is stored in the \code{pairwise_membership} slot if
\code{store_pairwise_membership = TRUE}.
\item Across all bioregionalizations, the function compares their pairwise
memberships to determine similarity. For each pair of bioregionalizations,
it computes a confusion matrix with the following elements:
}
\itemize{
\item \code{a}: Number of item pairs grouped in both bioregionalizations.
\item \code{b}: Number of item pairs grouped in the first but not in the second
bioregionalization.
\item \code{c}: Number of item pairs grouped in the second but not in the first
bioregionalization.
\item \code{d}: Number of item pairs not grouped in either bioregionalization.
}

The confusion matrix is stored in \code{confusion_matrix} if
\code{store_confusion_matrix = TRUE}.

Based on these confusion matrices, various indices can be computed to
measure agreement among bioregionalizations. The currently implemented
indices are:
\itemize{
\item \strong{Rand index}: \code{(a + d) / (a + b + c + d)}
Measures agreement by considering both grouped and ungrouped item pairs.
\item \strong{Jaccard index}: \code{a / (a + b + c)}
Measures agreement based only on grouped item pairs.
}

These indices are complementary: the Jaccard index evaluates clustering
similarity, while the Rand index considers both clustering and separation.
For example, if two bioregionalizations never group the same pairs, their
Jaccard index will be 0, but their Rand index may be > 0 due to ungrouped
pairs.

Users can compute additional indices manually using the list of confusion
matrices.

To identify which bioregionalization is most representative of the others,
the function can compute the correlation between the pairwise membership of
each bioregionalization and the total frequency of pairwise membership across
all bioregionalizations. This is enabled by setting \code{cor_frequency = TRUE}.
}
\examples{
# We here compare three different bioregionalizations
comat <- matrix(sample(0:1000, size = 500, replace = TRUE, prob = 1/1:1001),
20, 25)
rownames(comat) <- paste0("Site",1:20)
colnames(comat) <- paste0("Species",1:25)

dissim <- dissimilarity(comat, metric = "Simpson")
bioregion1 <- nhclu_kmeans(dissim, n_clust = 3, index = "Simpson")

net <- similarity(comat, metric = "Simpson")
bioregion2 <- netclu_greedy(net)
bioregion3 <- netclu_walktrap(net)

# Make one single data.frame with the bioregionalizations to compare
compare_df <- merge(bioregion1$clusters, bioregion2$clusters, by = "ID")
compare_df <- merge(compare_df, bioregion3$clusters, by = "ID")
colnames(compare_df) <- c("Site", "Hclu", "Greedy", "Walktrap")
rownames(compare_df) <- compare_df$Site
compare_df <- compare_df[, c("Hclu", "Greedy", "Walktrap")]

# Running the function
compare_bioregionalizations(compare_df)

# Find out which bioregionalizations are most representative
compare_bioregionalizations(compare_df,
                            cor_frequency = TRUE)
                                
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a5_2_compare_bioregionalizations.html}.

Associated functions:
\link{bioregionalization_metrics}
}
\author{
Boris Leroy (\email{leroy.boris@gmail.com}) \cr
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com})
}
