#' Buttler-Fickel Distance Matrix
#'
#' Computes a distance matrix following Buttler & Fickel (1995) for mixed-scale
#' variables. Each variable-specific distance matrix is normalized by its mean
#' pairwise distance (Gini mean difference), ensuring equal contribution of all
#' variables to the overall distance.
#'
#' @param df A data.frame where rows are cases and columns are variables.
#' @param types A character vector of the same length as \code{ncol(df)},
#'   indicating the scale level of each variable. Allowed values are
#'   \code{"metric"}, \code{"ordinal"}, or \code{"nominal"}.
#'
#' @return An object of class \code{dist}.
#'
#' @importFrom stats as.dist
#'
#' @export
buttler_fickel_dist <- function(df, types) {
  stopifnot(ncol(df) == length(types))

  p <- ncol(df)

  # distance functions
  d_nominal <- function(x) {
    x <- as.factor(x)
    outer(x, x, FUN = "!=") * 1
  }

  d_ordinal <- function(x) {
    if (is.factor(x)) x <- as.numeric(x)
    r <- rank(x, ties.method = "average")
    abs(outer(r, r, "-"))
  }

  d_metric <- function(x) {
    x <- as.numeric(x)
    abs(outer(x, x, "-"))
  }

  mean_pairdist <- function(D) {
    mean(D[upper.tri(D)])
  }

  D_list <- vector("list", p)

  for (h in seq_len(p)) {
    x <- df[[h]]
    type <- types[h]

    Dh <- switch(type,
                 "nominal" = d_nominal(x),
                 "ordinal" = d_ordinal(x),
                 "metric"  = d_metric(x),
                 stop(paste("Unknown type:", type))
    )

    DG <- mean_pairdist(Dh)

    if (DG == 0 || is.na(DG)) {
      warning("Variable ", colnames(df)[h], " has zero variance and will be ignored.")
      next
    }

    D_list[[h]] <- Dh / DG
  }

  D_list <- D_list[!sapply(D_list, is.null)]
  if (length(D_list) == 0) stop("No variable with positive variance available.")

  D_total <- Reduce("+", D_list)
  as.dist(D_total)
}
