#######################################################
#	apc package
#	Bent Nielsen,  2 Jul 2025, version 3.0.0
#	Bent Nielsen,  7 May 2017, version 1.3.1
#	function to get indices of data and to generate sub sets of data
#######################################################
#	Copyright 2014-2023 Bent Nielsen
#	Nuffield College, OX1 1NF, UK
#	bent.nielsen@nuffield.ox.ac.uk
#
#	This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.
#######################################################

apc.get.index	<- function(apc.data.list)
# BN 29 jun 2025:  renamed as apc.data.list.subset											     
#	BN 15 Aug 2023: added APm, PAm cases for mixed frequency, temporarily called new.apc.data.list.subset 
#	BN 11 Apr 2016: check age1, per1, coh1
#	BN 4 Jan 2016:  check values added.
#					change coh1 for "AP" and "PA"	
#	BN 6 Feb 2015
#	function to get indices to keep track of estimation and labelling
#	for regular data array use age,coh coordinates
#	for mixed   data array use age,per coordinates
#	in:		apc.data.list
#	out:	list 
{	#	apc.get.index
	#########################
	#	get values
	response	<- apc.data.list$response
	dose		<- apc.data.list$dose
	data.format	<- apc.data.list$data.format	
	age1		<- apc.data.list$age1		
	per1		<- apc.data.list$per1		
	coh1		<- apc.data.list$coh1		
	unit		<- apc.data.list$unit		
	per.zero	<- apc.data.list$per.zero	
	per.max		<- apc.data.list$per.max
	time.adjust	<- apc.data.list$time.adjust
	label		<- apc.data.list$label
	n.decimal	<- apc.data.list$n.decimal
	#########################
	#	check values - 3 Jan 2016
	if(is.null(time.adjust))	time.adjust <- 0
	########################
	#	BN 7 Dec 2023
	#	types of data format
	data.format.list.regular<- c("AP","AC","CA","CL","CP","PA","PC","trap","trapezoid")
	#########################
	#	data.format SPECIFIC CODE
	#########################
	if(data.format=="AC")
	#	matrix with age/cohort as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		age.max		<- nrow
		coh.max		<- ncol
		per.max		<- age.max+coh.max-1
		per.zero	<- 0
		age1		<- age1
		coh1		<- coh1
		per1		<- age1+coh1-time.adjust
		index.data	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+ncol),1]	<- row		 	#	age
			index.data[(col+1):(col+ncol),2]	<- seq(1,ncol)	#	cohort
			col	<- col + ncol
		}
		index.trap	<- index.data
		data.xlab	<- "age"
		data.ylab	<- "cohort"
		data.x1	<- age1
		data.y1	<- coh1
	}
	#########################
	if(data.format=="AP")
	#	matrix with age/period as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		age.max		<- nrow
		per.max		<- ncol
		coh.max		<- age.max+per.max-1
		per.zero	<- age.max-1
		age1		<- age1
		per1		<- per1
		coh1		<- per1 - (per.zero*unit[1]+age1)+time.adjust				# 16 Apr 2016: corrected; 15 Aug 2023: updated
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+ncol),1]	<- row		  							#	age
			index.data[(col+1):(col+ncol),2]	<- seq(1,ncol)							#	period
			index.trap[(col+1):(col+ncol),1]	<- row									#	age
			index.trap[(col+1):(col+ncol),2]	<- seq((nrow-row+1),(nrow-row+ncol))	#	cohort
			col	<- col + ncol
		}
		data.xlab	<- "age"
		data.ylab	<- "period"
		data.x1	<- age1
		data.y1	<- per1
	}
	#########################
	if(data.format=="CA")
	#	matrix with cohort/age as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		age.max		<- ncol
		coh.max		<- nrow
		per.max		<- age.max+coh.max-1
		per.zero	<- 0
		age1		<- age1
		coh1		<- coh1
		per1		<- age1+coh1-time.adjust
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+ncol),1]	<- row		  							#	age
			index.data[(col+1):(col+ncol),2]	<- seq(1,ncol)							#	cohort
			index.trap[(col+1):(col+ncol),1]	<- seq(1,ncol)							#	cohort 
			index.trap[(col+1):(col+ncol),2]	<- row		  							#	age    
			col	<- col + ncol
		}
		data.xlab	<- "cohort"
		data.ylab	<- "age"
		data.x1	<- coh1
		data.y1	<- age1
	}
	#########################
	if(data.format=="CL")
	#	square matrix with cohort/age as increasing row/column index
	#	NA: in botton right triangle so period >= age.max+period.max
	{	k	<- min(nrow(response),ncol(response))
		##############################
		#	check obligatory input
		if(ncol(response) != nrow(response))	return(cat("apc.get.index error: Response matrix is not square \n"))
		for(age in 2:k)
			for(coh in (k+2-age):k)
				if(is.na(response[coh,age])==FALSE) return(cat("apc.get.index error: Lower triangle of response matrix should be NA \n"))
		##############################
		nrow		<- k
		ncol		<- k
		n.data		<- k*(k+1)/2
		age.max		<- k
		coh.max		<- k
		per.max		<- k
		per.zero	<- 0
		age1		<- age1
		coh1		<- coh1
		per1		<- age1+coh1-time.adjust
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+k+1-row),1]	<- row		  							#	cohort
			index.data[(col+1):(col+k+1-row),2]	<- seq(1,k+1-row)						#	age   
			index.trap[(col+1):(col+k+1-row),1]	<- seq(1,k+1-row)						#	age    
			index.trap[(col+1):(col+k+1-row),2]	<- row		  							#	cohort 
			col	<- col +k+1-row
		}
		data.xlab	<- "underwriting time (cohort)"
		data.ylab	<- "development time (age)"
		data.x1	<- coh1
		data.y1	<- age1
	}
	#########################
	if(data.format=="CP")
	#	BN 25 Aug 2023: note, not used for mixed case, so unit=c(*,1,1).
	#				update if generalizing mixed case
	#	matrix with cohort/period as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		coh.max		<- nrow
		per.max		<- ncol
		age.max		<- coh.max+per.max-1
		per.zero	<- coh.max-1
		per1		<- per1
		coh1		<- coh1
		age1		<- per1 - (per.zero*unit[1]+coh1)+time.adjust 						
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+ncol),1]	<- row		  							#	cohort
			index.data[(col+1):(col+ncol),2]	<- seq(1,ncol)							#	period
			index.trap[(col+1):(col+ncol),1]	<- seq((nrow-row+1),(nrow-row+ncol))	#	age
			index.trap[(col+1):(col+ncol),2]	<- row									#	cohort
			col	<- col + ncol
		}
		data.xlab	<- "cohort"
		data.ylab	<- "period"
		data.x1	<- coh1
		data.y1	<- per1
	}
	#########################
	if(data.format=="PA")
	#	BN 25 Aug 2023: note, not used for mixed case, so unit=c(*,1,1).
	#				update if generalizing mixed case
	#	matrix with period/age as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		age.max		<- ncol
		per.max		<- nrow
		coh.max		<- age.max+per.max-1
		per.zero	<- age.max-1
		age1		<- age1
		per1		<- per1
		coh1		<- per1 - (per.zero*unit[1]+age1)+time.adjust						# 16 Apr 2016: corrected, 23 Apr 2023 unit
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		row	<- 0
		for(col in 1:ncol)
		{
			index.data[(row+1):(row+nrow),1]	<- seq(1,nrow)		  					#	period         
			index.data[(row+1):(row+nrow),2]	<- col									#	age            
			index.trap[(row+1):(row+nrow),1]	<- col									#	age
			index.trap[(row+1):(row+nrow),2]	<- seq((ncol-col+1),(ncol-col+nrow))	#	cohort
			row	<- row + nrow
		}
		data.xlab	<- "period"
		data.ylab	<- "age"
		data.x1	<- per1
		data.y1	<- age1
	}
	#########################
	if(data.format=="PC")
	#	matrix with period/cohort as increasing row/column index
	#	BN 25 Aug 2023: note, not used for mixed case, so unit=c(*,1,1).
	#				update if generalizing mixed case
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		coh.max		<- ncol
		per.max		<- nrow
		age.max		<- coh.max+per.max-1
		per.zero	<- coh.max-1
		per1		<- per1
		coh1		<- coh1
		age1		<- per1 - (per.zero*unit[1]+coh1)+time.adjust
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		row	<- 0
		for(col in 1:ncol)
		{
			index.data[(row+1):(row+nrow),1]	<- seq(1,nrow)							#	period
			index.data[(row+1):(row+nrow),2]	<- col									#	cohort
			index.trap[(row+1):(row+nrow),1]	<- seq((ncol-col+1),(ncol-col+nrow))	#	age
			index.trap[(row+1):(row+nrow),2]	<- col									#	cohort
			row	<- row + nrow
		}
		data.xlab	<- "period"
		data.ylab	<- "cohort"
		data.x1	<- per1
		data.y1	<- coh1
	}
	#########################
	if(data.format %in% c("trap","trapezoid"))
	# BN 1 Jul 2025: now also allow trap
	#	BN 25 Aug 2023: note, not used for mixed case, so unit=c(*,1,1).
	#	trapezoid matrix with age/cohort as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		age.max		<- nrow
		per.max		<- per.max
		coh.max		<- ncol
		per.zero	<- per.zero
		dim.lower	<- coh.max + age.max - 1 -per.zero - per.max		#	dimension of lower right triangle
		n.data		<- nrow*ncol - per.zero*(per.zero+1)/2 - dim.lower*(dim.lower+1)/2
		age1		<- age1
		coh1		<- coh1
		per1		<- age1+coh1+per.zero*unit[1]-time.adjust				#	Changed 10 April 2016
		index.data	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(age in 1:age.max)
		{
			col.zero	<- max(per.zero-age+1,0)
			col.max		<- min(coh.max,per.zero+per.max+1-age)
			index.data[ (col+1):(col+col.max-col.zero),1]	<- age					    #	age
			index.data[ (col+1):(col+col.max-col.zero),2]	<- seq(col.zero+1,col.max)	#	cohort
			col	<- col + col.max-col.zero
		}
		index.trap	<- index.data
		data.xlab	<- "age"
		data.ylab	<- "cohort"
		data.x1	<- age1
		data.y1	<- coh1
	}
	#########################
	if(data.format=="APm")
	#	7 Dec 2023
	#	mixed frequency APm
	#	matrix with age/period as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		age.max		<- nrow
		per.max		<- ncol
#		coh.max		<- age.max*unit[2]+(per.max-1)*unit[3]
		coh.max		<- (age.max-1)*unit[2]+per.max*unit[3]
		per.zero	<- age.max-1
		age1		<- age1
		per1		<- per1
		coh1		<- per1 - (age.max-1)*unit[1]*unit[2] - age1 +time.adjust
		index.data	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+ncol),1]	<- row		 	#	age
			index.data[(col+1):(col+ncol),2]	<- seq(1,ncol)	#	period
			col	<- col + ncol
		}
		index.trap	<- index.data
		data.xlab	<- "age"
		data.ylab	<- "period"
		data.x1	<- age1
		data.y1	<- per1
####		G	<- unit[2];
####		H	<- unit[3];
####		v.g	<- age.max-index.trap[row,1]		#	g, so that age=A-g*G 
####		v.h	<- index.trap[row,2]-1				#	h, so that per=H+h*H 
####		v.q.g <- g %/% H						#	Euclidean representation		
####		v.r.g	<- g %% H						#   g = v.g*G+r.g 
####		v.q.h	<- h %/% G
####		v.r.h <- h %% G					
	}
	#########################
	if(data.format=="PAm")
	#	16 Aug 2023
	#	mixed frequency PA
	#	matrix with period/age as increasing row/column index
	#	NA: none
	{	nrow		<- nrow(response)
		ncol		<- ncol(response)
		n.data		<- nrow*ncol
		age.max		<- ncol
		per.max		<- nrow
#		coh.max		<- age.max*unit[2]+(per.max-1)*unit[3]
		coh.max		<- (age.max-1)*unit[2]+per.max*unit[3]
		per.zero	<- age.max-1
		age1		<- age1
		per1		<- per1
		coh1		<- per1 - (age.max-1)*unit[1]*unit[2] - age1  +time.adjust
		index.data	<- matrix(nrow=n.data,ncol=2)
		index.trap	<- matrix(nrow=n.data,ncol=2)
		col	<- 0
		for(row in 1:nrow)
		{
			index.data[(col+1):(col+ncol),1]	<- row		  							#	age
			index.data[(col+1):(col+ncol),2]	<- seq(1,ncol)							#	period
			index.trap[(col+1):(col+ncol),1]	<- seq(1,ncol)							#	period 
			index.trap[(col+1):(col+ncol),2]	<- row		  							#	age    
			col	<- col + ncol
		}
		data.xlab	<- "period"
		data.ylab	<- "age"
		data.x1	<- per1
		data.y1	<- age1
####		G		<- unit[2];
####		H		<- unit[3];
####		v.g		<- age.max-index.trap[row,2]		#	g, so that age=A-g*G 
####		v.h		<- index.trap[row,1]-1				#	h, so that per=H+h*H 
####		v.q.g 	<- g %/% H						#	Euclidean representation		
####		v.r.g	<- g %% H						#   g = v.g*G+r.g 
####		v.q.h	<- h %/% G
####		v.r.h 	<- h %% G					
	}
	#########################
	#	GENERAL CODE
	#########################
	#	get anchoring	
####	if(data.format %in% data.format.list.regular)	{	# BN 12 Dec 2023
		if(per.zero %% 2==0)	{	U <- (per.zero+2)%/% 2;	per.odd <- FALSE;	} 
		else					{	U <- (per.zero+3)%/% 2;	per.odd <- TRUE; 	}
####	}
####		if(data.format %in% data.format.list.regular)	{	# BN 17 Dec 2023
####			G	<- H	<- 1;		
####			v.g	<- v.h	<- v.q.g	<- v.r.g	<- v.q.h	<- v.r.h	<- NULL
####		}	
	########################
	list.to.be.returned	<- list(
	  response	  =response	  ,	 #	argument
		dose		    =dose	    	,	 #	argument
		data.format	=data.format,	 #	argument
		unit		    =unit		    ,	 #	argument
		data.xmax	  =nrow	    	,
		data.ymax	  =ncol	    	,
		data.xlab	  =data.xlab	,
		data.ylab	  =data.ylab 	,
		data.x1		  =data.x1	  ,
		data.y1		  =data.y1  	,
		n.data		  =n.data		  ,
		index.data	=index.data	,
		index.trap	=index.trap	,
		age.max	  	=age.max	  ,
		per.max		  =per.max	  ,
		coh.max	  	=coh.max  	,
		per.zero	  =per.zero	  ,
		per.odd	  	=per.odd  	,
		U			      =U	    		,
		age1		    =age1	    	,
		per1    		=per1		    ,
		coh1		    =coh1   	 	,
#		G			=G			,			
#		H			=H			,
#		v.g			=v.g		,	 
#		v.h			=v.h		,	 
#		v.q.g 		=v.q.g 		,
#		v.r.g		=v.r.g		,
#		v.q.h		=v.q.h		,
#		v.r.h 		=v.r.h 		,
		label		    =label	    ,
		n.decimal	  =n.decimal	)
	return(list.to.be.returned)
}	#	apc.get.index


apc.data.list.subset <- function (apc.data.list, age.cut.lower = 0, age.cut.upper = 0, 
											     per.cut.lower = 0, per.cut.upper = 0,
											     coh.cut.lower = 0, coh.cut.upper = 0,
											     apc.index = NULL, suppress.warning = FALSE)
# BN    29 jun 2025:  renamed as apc.data.list.subset											     
#	BN		16 aug 2023:	adapted for mixed frequency, temporarily called new.apc.data.list.subset
#	BN		24 apr 2017:	new code provided.
#							Jonas Harnau found bug in old code
#							& suggested completely new code
#							This code is improved here. 
#							The new code replaces code in version 1.3 from 1 Dec 2016 and earlier
#	function to get subset of data set
#	in:		apc.data.list
#	out:	list 
#	note:	if apc.index supplied then it suffices to input
#			apc.data.list = list(response=response,data.format=data.format,dose=dose)
#				where dose could be NULL
#			apc.index does not need to be a full apc.index list. Sufficient entries are
#						data.format
#						age.max
#						per.max
#						coh.max
#						index.trap
#						index.data
#						per.zero
#	note2: 	in code:
#			.cut refers to subset in old coordinate system
#			.new refers to subset in new coordinate system
{	#	apc.data.list.subset
	##############################
	#	data format lists
	data.format.list.regular<- c("AP","AC","CA","CL","CP","PA","PC","trap","trapezoid")
	data.format.list.mixed	<- c("APm","PAm")
	######################
	#	get index
  	if(is.null(apc.index)) apc.index <- apc.get.index(apc.data.list)
	##############################
	#	get index values, that are used
	data.format	<- apc.index$data.format
	age.max		  <- apc.index$age.max				
	per.max	  	<- apc.index$per.max				
	coh.max 		<- apc.index$coh.max
	age1  		  <- apc.index$age1    
	per1	    	<- apc.index$per1    
	coh1	  	  <- apc.index$coh1
	unit  		  <- apc.index$unit
	per.zero	  <- apc.index$per.zero
	index.data	<- apc.index$index.data
	index.trap	<- apc.index$index.trap
	##############################
	#	get data.list values, that are used
	response	  <- apc.data.list$response
	dose		    <- apc.data.list$dose
	##############################
	#	warnings
	if(age.cut.lower>age.max-3 & !suppress.warning)	warning("age.cut.lower >= age.dim-2")
	if(per.cut.lower>per.max-3 & !suppress.warning)	warning("per.cut.lower >= per.dim-2")
	if(coh.cut.lower>coh.max-3 & !suppress.warning)	warning("coh.cut.lower >= coh.dim-2")
	if(age.cut.upper>age.max-3 & !suppress.warning)	warning("age.cut.upper >= age.dim-2")
	if(per.cut.upper>per.max-3 & !suppress.warning)	warning("per.cut.upper >= per.dim-2")
	if(coh.cut.upper>coh.max-3 & !suppress.warning)	warning("coh.cut.upper >= coh.dim-2")
	##############################
	#	errors
	if(age.cut.lower>=age.max)	stop("age.cut.lower >= age.dim")
	if(per.cut.lower>=per.max)	stop("per.cut.lower >= per.dim")
	if(coh.cut.lower>=coh.max)	stop("coh.cut.lower >= coh.dim")
	if(age.cut.upper>=age.max)	stop("age.cut.upper >= age.dim")
	if(per.cut.upper>=per.max)	stop("per.cut.upper >= per.dim")
	if(coh.cut.upper>=coh.max)	stop("coh.cut.upper >= coh.dim")
	##############################
	#	Regular data format
	if(data.format %in% data.format.list.regular)
	{
		##############################
		#	paste per=age+coh-1 column onto index.trap
		index.trap	<-	cbind(index.trap, apply(index.trap,1,sum)-1)
		##############################
		#	form binary vector with 1 if constraints are satisfied
		select	<-	( 
		        index.trap[,1] >  age.cut.lower                    
					& index.trap[,1] <= age.max - age.cut.upper
					& index.trap[,2] >  coh.cut.lower                    
					& index.trap[,2] <= coh.max - coh.cut.upper
					& index.trap[,3] >  per.cut.lower + per.zero                    
					& index.trap[,3] <= per.max - per.cut.upper	+ per.zero
					)
		#  if(nrow(index.match.cut) == 0) stop("Cuts produce empty data set.", call. = TRUE)
		##############################
  	  	#	new definitions of parameters of index array
		age.min.cut	<- min(index.trap[select,1])
		per.min.cut	<- min(index.trap[select,3])
		coh.min.cut	<- min(index.trap[select,2])
		age.max.cut	<- max(index.trap[select,1]) #-age.min.cut+1
		per.max.cut	<- max(index.trap[select,3]) #-per.min.cut+1
		coh.max.cut	<- max(index.trap[select,2]) #-coh.min.cut+1
		#	within new coordinate system
		age.max.new	<- age.max.cut-age.min.cut+1	
		per.max.new	<- per.max.cut-per.min.cut+1	
		coh.max.new	<- coh.max.cut-coh.min.cut+1
		per.zero.new<- max(0, per.min.cut -1 - age.min.cut - coh.min.cut + 2)
		age1.new    <- age1 + (age.min.cut -1) * unit[1] 	# 16 Aug 2023 updated for mixed
   	per1.new    <- per1 + (per.min.cut -1) * unit[1] 	# 16 Aug 2023 updated for mixed
  	coh1.new    <- coh1 + (coh.min.cut -1) * unit[1]	# 16 Aug 2023 updated for mixed
		##############################
		#	adjust coordinates in index.trap
		index.trap[,1]	<- index.trap[,1] - age.min.cut + 1
		index.trap[,2]	<- index.trap[,2] - coh.min.cut + 1
		##############################
  	#	new data
		dose.new	<- response.new	<- matrix(NA,nrow=age.max.new,ncol=coh.max.new)
		response.new[index.trap[select,c(1,2)]]	<- response[index.data[select,]]
		if(!is.null(dose)){
			dose.new[index.trap[select,c(1,2)]]	<- dose[index.data[select,]]
		}else dose.new = NULL
		##############################
		#	warnings
		# BN 2 July 2025 reorganized: 
		# suppress.warning only checked when cat/print warnings
		#########
		#	subset becomes too small
	  cut.old	<- c( age.cut.lower,age.cut.upper,
					        per.cut.lower,per.cut.upper,
					        coh.cut.lower,coh.cut.upper)
		cut.new <- c( age.min.cut-1,age.max-age.max.cut,
					        per.min.cut-1-per.zero,per.max-per.max.cut+per.zero,
					        coh.min.cut-1,coh.max-coh.max.cut)
		if(sum(cut.new)>sum(cut.old)){
		  if(!suppress.warning){
				cat("WARNING apc.data.list.subset: ")
				cat("cuts in arguments are:\n")
				print(cut.old)
				cat("have been modified to:\n")
				print(cut.new)
			}	
		}	
		#	data format is changed
	  if(  data.format %in% c("AP", "CA", "CL", "CP", "PA", "PC") ){
			cat("WARNING apc.data.list.subset: ")
			cat('coordinates changed to "AC"\n')
	  }  
	  if(  data.format %in% c("AC", "AP", "CA", "CL", "CP", "PA", "PC") ){
			cat("WARNING apc.data.list.subset: ")
			cat('data.format changed to "trapezoid"\n')
		}  
		if(!(data.format %in% c("trap","trapezoid") )){
			data.format	<- "trapezoid"
		}	
		##############################
		#	warnings
		#	age.max.cut: largest age value
		#	age.min.cut: smallest age value
		if(age.max.cut-age.min.cut<=1 & !suppress.warning)	warning("age dimension <= 2")
		if(per.max.cut-per.min.cut<=1 & !suppress.warning)	warning("per dimension <= 2")
		if(coh.max.cut-coh.min.cut<=1 & !suppress.warning)	warning("coh dimension <= 2")
	}	
	##############################
	#	16 Aug 2023
	#	Mixed data format
	if(data.format %in% data.format.list.mixed)
	{
		##############################
		#	warnings
		if(coh.cut.lower>0 & !suppress.warning)	warning("coh.cut.lower not used for mixed data format")
		if(coh.cut.upper>0 & !suppress.warning)	warning("coh.cut.upper not used for mixed data format")
		##############################
		#	form selection vectors
		v.age.sel	<- (1+age.cut.lower):(age.max-age.cut.upper)
		v.per.sel	<- (1+per.cut.lower):(per.max-per.cut.upper)
		if(data.format=="APm"){	v.row.sel	<- v.age.sel;	v.col.sel	<- v.per.sel	}
		if(data.format=="PAm"){	v.row.sel	<- v.per.sel;	v.col.sel	<- v.age.sel	}
		##############################
		#	warnings
		#	age.max.cut: largest age value
		#	age.min.cut: smallest age value
		if(length(v.age.sel)<=2 & !suppress.warning)	warning("age dimension <= 2")
		if(length(v.per.sel)<=2 & !suppress.warning)	warning("per dimension <= 2")
#		##############################
#		#	form binary vector with 1 if constraints are satisfied
#		select	<-	( index.trap[,1] >  age.cut.lower                    
#					& index.trap[,1] <= age.max - age.cut.upper
#					& index.trap[,2] >  per.cut.lower                    
#					& index.trap[,2] <= per.max - per.cut.upper
#					)
		##############################
  	  	#	new definitions of parameters of index array
		age.max.new	<- age.max - age.cut.upper - age.cut.lower			
		per.max.new	<- per.max - per.cut.upper - per.cut.lower
		age1.new  	<- age1 + age.cut.lower	* unit[1] * unit[2]
		per1.new	  <- per1 + per.cut.lower	* unit[1] * unit[3]
		coh1.new	  <- NULL
		per.zero.new<- NULL
#		##############################
#		#	adjust coordinates in index.data
#		index.data.new		<- index.data
#		index.data.new[,1]	<- index.data[,1] - age.cut.lower
#		index.data.new[,2]	<- index.data[,2] - per.cut.lower
		##############################
  	  	#	new data
		response.new	<- response[v.row.sel,v.col.sel]
		if(is.null(dose))	dose.new = NULL
		else   dose.new <-	   dose[v.row.sel,v.col.sel]		
#		if(data.format=="APm")
#			dose.new	<- response.new	<- matrix(NA,nrow=age.max.new,ncol=per.max.new)
#		if(data.format=="PAm")
#			dose.new	<- response.new	<- matrix(NA,nrow=per.max.new,ncol=age.max.new)
#		response.new[index.data.new[select,]]	<- response[index.data[select,]]
#		if(is.null(dose))	dose.new = NULL
#		else dose.new[index.data.new[select,]]	<- 	   dose[index.data[select,]]
	}	# mixed frequency
	##############################
	return(list(
	      response	  =response.new				      ,
				dose    		=dose.new					        ,
				data.format	=data.format				      ,
				age1		    =age1.new					        ,
				per1    		=per1.new					        ,
				coh1		    =coh1.new					        ,
				unit    		=apc.data.list$unit			  ,
				per.zero  	=per.zero.new				      ,
				per.max		  =per.max.new				      ,
				time.adjust	=apc.data.list$time.adjust,
				label   		=apc.data.list$label		  ,
				n.decimal	  =apc.data.list$n.decimal	))
}	#	apc.data.list.subset

			

			