#' Run a demo of alpmixBayes
#'
#' This runs alpmixBayes on the packaged example dataset ewm1.100.
#' @param verbose Logical. If TRUE, progress messages are printed. Default is TRUE.
#' @return Summary of the Bayesian mixture model
#' @export
#' @examples
#' \donttest{
#' # Run the demo (may take a few moments)
#' demo_run()
#' }

demo_run <- function(verbose = TRUE) {
  # Explicitly get the dataset from the package namespace
  ewg_data <- get("ewg", envir = asNamespace("alpmixBayes"))

  # Check if data was successfully loaded
  if (is.null(ewg_data)) {
    stop("Could not load 'ewg' dataset from alpmixBayes package")
  }

  # Extract the y matrix from the first element
  y_matrix <- ewg_data[[1]]$y

  # Convert the matrix to a vector and use first 100 observations
  d <- as.vector(y_matrix)[1:100]

  # Conditional messaging
  if (verbose) {
    message("Data vector length: ", length(d))
  }

  # MCMC, initial values, and priors
  mcmc_values <- list(nburn = 50, niter = 100, thin = 2)
  init_values <- list(alpha=1, p=0.5, p1=0.3, p2=0.3,
                      th1=1.5, th2=1.5, th3=2)
  prior <- list(ap=c(1,0.001), pp=c(1,1,1),
                th1p=c(1,1), th2p=c(1,1), th3p=c(1,1))

  # Run the model
  ptm <- proc.time()
  r <- alpmixBayes(d, mcmc_values, init_values, prior, survmodel = "EWG")
  runtime <- proc.time() - ptm

  if (verbose) {
    message("Runtime: ", round(runtime[3], 2), " seconds")
  }

  # Return the summary object WITHOUT printing it
  summary_result <- summary(r)

  # Only print if verbose is TRUE
  if (verbose) {
    print(summary_result)
  }

  # Return invisibly so it doesn't auto-print
  return(invisible(summary_result))
}
