## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>", 
  message = FALSE, 
  warning = FALSE
)

## -----------------------------------------------------------------------------
data(chickwts)
chickwts$feed <- reorder(chickwts$feed, 
                         chickwts$weight, 
                         FUN=mean)
chick_mod <- lm(weight~ feed, data=chickwts)

## -----------------------------------------------------------------------------
library(marginaleffects)
library(VizTest)
## make predictions 
chick_preds <- predictions(chick_mod, variables="feed", by="feed")

## save predicted values in chick_b
chick_b <- coef(chick_preds)

## set names of predicted values
names(chick_b) <- chick_preds$feed

## make into visual testing data
chick_vt_data <- make_vt_data(est=chick_b, vcov(chick_preds))

## execute viztest function on predictions
chick_vt <- viztest(chick_vt_data, include_zero=FALSE)

## -----------------------------------------------------------------------------
chick_annots <- make_annotations(chick_vt)
chick_annots

## ----chick-sig, fig.width=6, fig.height=6, fig.align="center", out.width="75%", fig.cap="Predicted Chick Weights with 95% Confidence Intervals and Insignificance Brackets.  The estimates in pairs marked 'NS' are not significantly different from each other.  All other pairs are statistically different from each other whether the intervals overlap or not."----
library(ggplot2)
library(ggsignif)
ggplot(chick_preds, aes(x=feed, y=estimate)) +
  geom_pointrange(aes(ymin=conf.low, ymax=conf.high)) +
  do.call(geom_signif, chick_annots) +
  labs(
    x = "Feed Type",
    y = "Predicted Weight"
  ) + 
  theme_bw()

## -----------------------------------------------------------------------------
## Load Data
data(Ornstein, package="carData")

## Estimate Model
orn_mod <- glm(interlocks ~ log2(assets) + sector + nation, data=Ornstein, 
    family=poisson)

## Generate Predictions
orn_preds <- predictions(orn_mod, variables = "sector", by = "sector")

## ----orn-annot1, fig.height=6, fig.width=6, out.width="75%", fig.align="center", fig.cap="Predicted Number of Interlocks by Sector with 95% Confidence Intervals and Significance Brackets.  The estimates in pairs marked with asterisks are significantly different from each other.  All other overlapping pairs are not significantly different from each other."----
orn_b <- coef(orn_preds)
names(orn_b) <- orn_preds$sector
orn_vt_data <- make_vt_data(est=orn_b, vcov(orn_preds))
orn_vt <- viztest(orn_vt_data, include_zero=FALSE)
orn_annots <- make_annotations(orn_vt, adjust="none")

ggplot(orn_preds, aes(x=sector, y=estimate)) +
  geom_pointrange(aes(ymin=conf.low, ymax=conf.high)) +
  do.call(geom_signif, orn_annots) +
  labs(
    x = "Sector",
    y = "Predicted Number of Interlocks"
  ) + 
  theme_bw()

## ----orn-annot2, fig.height=6, fig.width=6, out.width="75%", fig.align="center", fig.cap="Predicted Number of Interlocks by Sector with 95% Confidence Intervals and Nudged Significance Brackets."----
orn_annots <- make_annotations(orn_vt, adjust="none", 
                               nudge=c(2, 1, 2, 0,0) )

ggplot(orn_preds, aes(x=sector, y=estimate)) +
  geom_pointrange(aes(ymin=conf.low, ymax=conf.high)) +
  do.call(geom_signif, orn_annots) +
  labs(
    x = "Sector",
    y = "Predicted Number of Interlocks"
  ) + 
  theme_bw()

