#' @importFrom plotly layout
#' @importFrom plotly subplot
#' @importFrom stringr %>%

#' @title Plot multiple core sets
#' @description Given multiple games with two, three or four players, this function draws in a single plot their projected core sets and sets of imputations.
#' @param games A matrix in which each row is a characteristic function.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if the characteristic functions in \code{games} are introduced in binary order instead of lexicographic order.
#' @param imputations A logical value. By default, \code{imputations=TRUE}.
#'                    When set to \code{imputations=FALSE}, the sets of imputations are not drawn.
#' @return A plot of the given core sets.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @examples
#' # Plotting the core and the least core of a game:
#' v <- c(0,0,0,0,10,40,30,60,10,20,90,90,90,130,160)
#' vt <- leastcore(v)$vt
#' plotcoresets(games = rbind(v,vt), imputations = FALSE)
#' @seealso \link{plotcoreset}
#' @export

plotcoresets <- function(games, binary = FALSE, imputations = TRUE) {

  if (dim(games)[2] %in% c(3,7,15) == FALSE) {
    stop("The rows in 'games' do not represent characteristic functions of games with two, three or four players.")
  }

  if (length(imputations) == 1) {
    imputations <- rep(imputations, dim(games)[1])
  }

  colors <- c("blue", "red", "green", "cyan", "yellow")
  while (length(colors) < dim(games)[1]) {
    colors <- c(colors, rep(c("blue", "red", "green", "cyan", "yellow")))
  }

  labels <- paste("Core", 1:dim(games)[1])

  # Fabricamos la gráfica con el primer juego:
  fig <- subplot(plotcoresetsaux(v = games[1,], imputations = imputations[1], projected = TRUE, color = colors[1]))

  # Ahora vamos metiendo los demás juegos en la misma gráfica:
  for (i in 2:dim(games)[1]) {
    fig <- suppressWarnings(subplot(fig, plotcoresetsaux(v = games[i,], imputations = imputations[i], projected = TRUE, color = colors[i])))
  }

  # Create manual legend as annotations
  legend_annotations <- list()
  for (i in 1:dim(games)[1]) {
    legend_annotations[[i]] <- list(
      x = 1,  # Position outside the plot
      y = 1.05 - (i * 0.05),  # Stack labels vertically
      text = paste(labels[i]),
      showarrow = FALSE,
      font = list(color = colors[i], size = 14),
      xref = "paper",
      yref = "paper"
    )
  }

  # Add layout
  fig <- fig %>% layout(
    title = list(text = "multiple core sets", font = list(color = "black")),
    margin = list(t = 50),
    showlegend = FALSE,  # Hide default legend
    scene = list(
      xaxis = list(title = "Player 1"),
      yaxis = list(title = "Player 2"),
      zaxis = list(title = "Player 3")
    ),
    annotations = legend_annotations  # Add manual legend
  )

  suppressWarnings(print(fig))
}
