#' TBA
#'
#' @description
#' Calculate the targeted biomass adjustment (TBA), which dampens the influence
#' of proportional rate of change \eqn{rb} on catch advice when the proportion of abundance
#' targeted by harvest slot limits is small. Optionally, produce a graph showing how
#' TBA varies across a grid of (\code{P_targeted}, \code{rb}) values.
#'
#' The TBA is calculated as \eqn{1 + (P_{\mathrm{targeted}} \times rb)}.
#'
#' @param P_targeted Numeric (length 1) in \[0, 1]. Proportion of abundance targeted
#'   by harvest slot limits (e.g., from \code{prop_target()}).
#' @param rb Numeric (length 1). Proportional rate of change in a biomass index
#'   (e.g., from \code{rb()}). Typical values lie in \[-1, 1], but larger magnitudes are allowed.
#' @param digits Integer. Number of decimal places used to round outputs
#'   (default = 2). Set \code{digits = NA} to prevent rounding.
#' @param plot Logical. If \code{TRUE}, include a \pkg{ggplot2} plot of the calculated value
#'   on a grid of (\code{P_targeted}, \code{rb}) combinations (default \code{FALSE}).
#'
#' @return A list with:
#' \describe{
#'   \item{P_targeted}{Input targeted proportion (numeric scalar).}
#'   \item{rb}{Input proportional rate of change (numeric scalar).}
#'   \item{damped_change}{\eqn{P_{\mathrm{targeted}} \times rb} (numeric scalar).}
#'   \item{TBA}{Targeted biomass adjustment multiplier \eqn{1 + P_{\mathrm{targeted}} \times rb}
#'              (numeric scalar). \code{TBA > 1} increases advised catch; \code{TBA < 1} decreases it.}
#'   \item{plot}{(only when \code{plot=TRUE}) a \pkg{ggplot2} object visualizing TBA over a grid.}
#' }
#'
#' @details
#' The adjustment dampens large changes in biomass indices when the targeted
#' proportion is small by multiplying \code{rb} by \code{P_targeted}. The plot shades
#' the surface of \eqn{1 + P \times rb}. A dashed horizontal line marks \code{rb = 0}.
#'
#' @examples
#' TBA(P_targeted = 0.5, rb = -0.5)               # compute only
#' \donttest{
#' # compute + plot (requires ggplot2)
#' out <- TBA(P_targeted = 0.5, rb = -0.5, digits = 2, plot = TRUE)
#' out$plot
#' }
#'
#' @seealso \code{\link{prop_target}} for targeted proportion; \code{\link{rb}} for proportional rate of change.
#' @export
TBA <- function(P_targeted = NULL, rb = NULL, digits = 2, plot = FALSE) {
  # ---- validation ----
  nums <- list(P_targeted = P_targeted, rb = rb)
  bad_len <- vapply(nums, function(x) !is.numeric(x) || length(x) != 1L || !is.finite(x), logical(1))
  if (any(bad_len)) stop("`P_targeted` and `rb` must be finite, numeric scalars.", call. = FALSE)

  if (P_targeted < 0 || P_targeted > 1)
    stop("`P_targeted` must be in [0, 1].", call. = FALSE)

  if (!is.null(digits) && !is.na(digits)) {
    if (length(digits) != 1 || !is.numeric(digits) || digits < 0 || digits %% 1 != 0)
      stop("`digits` must be a single non-negative integer or NA.", call. = FALSE)
  }

  if (!is.logical(plot) || length(plot) != 1L || is.na(plot))
    stop("`plot` must be a single logical (TRUE/FALSE).", call. = FALSE)

  if (abs(rb) > 1) {
    warning("`rb` is typically in [-1, 1]; values outside this range indicate very large changes.", call. = FALSE)
  }

  # ---- compute components ----
  damped_change <- P_targeted * rb
  multiplier <- 1 + damped_change

  # ---- rounding policy ----
  if (!is.null(digits) && !is.na(digits)) {
    damp_out <- round(damped_change, digits)
    tba_out  <- round(multiplier, digits)
  } else {
    damp_out <- damped_change
    tba_out  <- multiplier
  }

  out <- list(
    P_targeted    = P_targeted,
    rb            = rb,
    damped_change = damp_out,
    TBA           = tba_out
  )

  # ---- optional plot ----
  if (isTRUE(plot)) {
    if (!requireNamespace("ggplot2", quietly = TRUE))
      stop("Package 'ggplot2' is required for plotting.", call. = FALSE)

    .use_linewidth <- function() {
      requireNamespace("ggplot2", quietly = TRUE) &&
        utils::packageVersion("ggplot2") >= "3.4.0"
    }

    # Fixed grid resolution
    step <- 0.01

    # Build plotting ranges; expand rb range to include the input if needed
    rb_min <- min(-1, floor(rb * 10) / 10)
    rb_max <- max( 1, ceiling(rb * 10) / 10)

    P_range <- seq(0, 1, by = step)
    R_range <- seq(rb_min, rb_max, by = step)

    # Ensure at least two points per axis
    if (length(P_range) < 2L) P_range <- c(0, 1)
    if (length(R_range) < 2L) R_range <- sort(c(-1, 0, 1))

    grid <- expand.grid(P = P_range, R = R_range, KEEP.OUT.ATTRS = FALSE, stringsAsFactors = FALSE)
    grid$Mult <- 1 + grid$P * grid$R

    # symmetric colour limits around 1 for a balanced palette
    max_dev <- max(abs(grid$P * grid$R))
    zlim <- 1 + c(-max_dev, max_dev)
    if (!is.finite(max_dev) || max_dev == 0) zlim <- c(0.75, 1.25)

    p <- ggplot2::ggplot(grid, ggplot2::aes(x = P, y = R, fill = Mult)) +
      ggplot2::geom_tile(alpha = 0.9) +
      ggplot2::scale_fill_gradient2(
        name = "",
        low = "darkred", mid = "lightyellow", high = "darkgreen",
        midpoint = 1, limits = zlim
      ) +
      ggplot2::guides(
        fill = ggplot2::guide_colorbar(
          title = NULL,
          direction = "horizontal",
          barwidth = grid::unit(0.3, "npc"),
          barheight = grid::unit(10, "pt"),
          label.position = "bottom",
          ticks = FALSE
        )
      ) +
      # geom_hline
      ggplot2::geom_hline(
        yintercept = 0,
        color = "black",
        linetype = "dashed",
        size = if (.use_linewidth()) NULL else 0.4,
        linewidth = if (.use_linewidth()) 0.4 else NULL
      ) +
      ggplot2::geom_point(
        data = data.frame(P = P_targeted, R = rb),
        ggplot2::aes(x = P, y = R),
        inherit.aes = FALSE, color = "black", size = 2.8
      ) +
      ggplot2::labs(
        title = "TBA",
        x = "P",
        y = "rb"
      ) +
      ggplot2::coord_cartesian(xlim = range(P_range), ylim = range(R_range), expand = FALSE) +
      ggplot2::theme_bw() +
      ggplot2::theme(
        plot.title = ggplot2::element_text(hjust = 0.5),
        plot.margin = grid::unit(c(0.25, 0.25, 0.25, 0.25), "cm"),
        text = ggplot2::element_text(size = 15),
        legend.position = "top",
        legend.direction = "horizontal",
        legend.background = ggplot2::element_blank(),
        legend.key = ggplot2::element_blank()
      )

    out$plot <- p
  }

  out
}
