#' Plot the SCF Convergence Process for Multiple Rounds of Optimization
#'
#' An internal function plots the generated SCF convergence tibble
#' @param SCFData The tibble generated by OptiSCFMonitorAsWholeTibble() describing the SCF convergence process for multiple rounds of optimization(if any).
#' @param SCFconver A numeric vector showing the SCF convergence requirement read from the gaussian .log file.
#' @param BOT A numeric vector describing the starting optimization round for plotting
#' @param TOP A numeric vector describing the ending optimization round for plotting
#' @returns No return value, called for side effects
#' @importFrom tidyr pivot_longer
#' @importFrom dplyr mutate filter
#' @importFrom stringr str_replace
#' @importFrom magrittr %>%
#' @importFrom ggplot2 ggplot aes geom_line geom_hline theme_minimal facet_wrap
#' @export
#' @examples
#' library(dplyr)
#' library(stringr)
#' library(ggplot2)
#' library(tidyr)
#'
#' temp_dat <- OptiSCFMonitorAsWholeTibble(SCFMonitorExample())
#' MultipleRoundOptiSCFplotting(temp_dat[[1]],
#'   SCFconver = -log10(temp_dat[[2]]),
#'   BOT = 10,
#'   TOP = 15
#' )
#'
#' @name MultipleRoundOptiSCFplotting

utils::globalVariables(names = c("Cycle", "OptiCycle", "value", "valueT", "SCFdatType"), package = "SCFMonitor")

MultipleRoundOptiSCFplotting <- function(SCFData, SCFconver, BOT, TOP) {
  SCFData <- SCFData %>%
    tidyr::pivot_longer(!c(Cycle, OptiCycle),
      names_to = "SCFdatType",
      values_to = "valueT"
    ) %>%
    dplyr::mutate(value = as.numeric(stringr::str_replace(valueT, "D", "E"))) %>%
    dplyr::filter((OptiCycle >= BOT) & (OptiCycle <= TOP))

  ggplot2::ggplot(
    data = SCFData,
    mapping = ggplot2::aes(
      x = Cycle,
      y = -log10(abs(value)),
      color = SCFdatType
    )
  ) +
    ggplot2::geom_line() +
    ggplot2::geom_hline(
      yintercept = SCFconver - 2 + 0.03,
      color = "#F8766D",
      linewidth = 1.05
    ) +
    ggplot2::geom_hline(yintercept = SCFconver - 2, color = "#00BA38") +
    ggplot2::geom_hline(yintercept = SCFconver, color = "#C77CFF") +
    ggplot2::theme_minimal() +
    ggplot2::facet_wrap(~OptiCycle)
}
