#' Class ClickhouseResult
#'
#' Clickhouse's query results class.  This classes encapsulates the result of an SQL
#' statement (either \code{select} or not).
#'
#' @export
#' @keywords internal
setClass("ClickhouseResult",
  contains = "DBIResult",
  slots = list(
    sql = "character",
    env = "environment",
    conn = "ClickhouseConnection",
    ptr = "externalptr",
    Int64 = "character",
    toUTF8 = "logical"
  )
)

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbFetch", signature = "ClickhouseResult", definition = function(res, n = -1, ...) {
  if (length(n) > 1) stop("n must be integer")
  if (is.infinite(n)) n <- -1
  if (n != as.integer(n) || (n < 0 && n != -1)) {
    stop("n must be a positive integer, -1 or Inf")
  }
  ret <- fetch(res@ptr, n)
  ret <- convert_Int64(ret, res@Int64)

  if(res@toUTF8 == TRUE) ret <- encode_UTF(ret)

  return(ret)
})

#' @importFrom bit64 as.integer64
convert_Int64 <- function(df, Int64) {
  if (Int64 == "character") return(df)
  int64Types <- c('Int64', 'UInt64', 'Nullable(Int64)', 'Nullable(UInt64)')
  toConvert <- which(attr(df, 'data.type') %in% int64Types)
  if(length(toConvert) > 0){
  as_Int64 <- switch(Int64,
                      integer = as.integer,
                      numeric = as.numeric,
                      integer64 = as.integer64
                     )
  df[toConvert] <- suppressWarnings(lapply(df[toConvert], as_Int64))
  return(df)}else{
    return(df)
  }
}

encode_UTF <- function(df){
  toConvert <- which(attr(df, "data.type") %in% c("String", "FixedString", "Nullable(String)", "Nullable(FixedString)"))

  if(length(toConvert) > 0){
    df[toConvert] <- suppressWarnings(lapply(df[toConvert], function(x) enc2utf8(x)))
  }
  return(df)
}


#' @rdname ClickhouseResult-class
#' @export
setMethod("dbClearResult", "ClickhouseResult", definition = function(res, ...) {
  if (!validPtr(res@ptr)) {
    warning("Result has already been cleared.")
  } else {
    clearResult(res@ptr)
  }
  invisible(TRUE)
})

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbHasCompleted", "ClickhouseResult", definition = function(res, ...) {
  hasCompleted(res@ptr)
})

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbGetStatement", "ClickhouseResult", function(res, ...) {
  getStatement(res@ptr)
})

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbIsValid", "ClickhouseResult", function(dbObj, ...) {
  validPtr(dbObj@ptr)
})

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbGetRowCount", "ClickhouseResult", function(res, ...) {
  getRowCount(res@ptr)
})

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbGetRowsAffected", "ClickhouseResult", definition = function(res, ...) {
  getRowsAffected(res@ptr)
})

#' @rdname ClickhouseResult-class
#' @export
setMethod("dbColumnInfo", "ClickhouseResult", definition = function(res, ...) {
  df <- dbFetch(res, 0)
  data.frame(
    name = colnames(df),
    field.type = resultTypes(res@ptr),
    data.type = sapply(df, class)
  )
})
