#' Performance Metrics for RANSAC Models
#'
#' Calculates the root mean square error (RMSE), mean absolute error (MAE), and coefficient of determination (R²)
#' for a model fitted using the RANSAC algorithm.
#'
#' @param model A model fitted via RANSAC (`ransac_model` or `ransac_nls`).
#' @param data Data frame containing the model variables.
#'
#' @return A data frame with RMSE, MAE, and R² calculated on the inliers.
#'
#' @examples
#' set.seed(123)
#' D <- seq(10, 50, by = 5)
#' H <- seq(15, 30, length.out = length(D))
#' V <- 0.01 * D^2 * H + rnorm(length(D), sd = 5)
#' V[c(3, 7)] <- V[c(3, 7)] + 50  # add outliers
#' data <- data.frame(D = D, H = H, V = V)
#'
#' model <- ransac_nls(V ~ a * D^b * H^c, data = data,
#'                     start = list(a = 0.01, b = 2, c = 1),
#'                     n_min = 4, n_iter = 100, tol = 10)
#'
#' metrics_ransac(model, data)
#'
#' @export
#' @importFrom stats predict formula
metrics_ransac <- function(model, data) {
  formula_model <- formula(model)
  y_name <- all.vars(formula_model)[1]
  y_obs <- data[[y_name]]
  inliers <- attr(model, "inliers")
  data_in <- data[inliers, ]

  pred_ransac <- predict(model, newdata = data_in)
  y_inliers <- data_in[[y_name]]

  rmse_r <- sqrt(mean((y_inliers - pred_ransac)^2))
  mae_r  <- mean(abs(y_inliers - pred_ransac))
  r2_r   <- 1 - sum((y_inliers - pred_ransac)^2) / sum((y_inliers - mean(y_inliers))^2)

  return(data.frame(
    RMSE = rmse_r,
    MAE = mae_r,
    R2 = r2_r
  ))
}
