% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Gene_Quantile_CenIPWE.R
\name{Gene_Quantile_CenIPWE}
\alias{Gene_Quantile_CenIPWE}
\title{A low-level function for the generic optimization step in 
estimating Quanilte-optimal treatment regime for censored data}
\usage{
Gene_Quantile_CenIPWE(data_aug, tau, p_level, regimeClass,
  cluster = FALSE, s.tol = 1e-04, it.num = 8, pop.size = 5000,
  Domains = NULL, sign_beta1 = NULL, Penalty.level = 0)
}
\arguments{
\item{data_aug}{a data.frame of the observed data after preprocessing. It should include be
augmented with a new column: \code{epsi} for the composite weights.}

\item{tau}{a value between 0 and 1. This is the quantile of interest.}

\item{p_level}{choose between 0,1,2,3 to indicate different levels of output
from the genetic function. Specifically, 0 (minimal printing),
  1 (normal), 2 (detailed), and 3 (debug).}

\item{regimeClass}{a formula specifying the class of treatment regimes to search,
e.g. if \code{regimeClass = a~x1+x2}, and then this function will 
search the class of treatment regimes
of the form 
\deqn{d(x) = I \left(\beta_0 +\beta_1  x_1 + \beta_2  x_2 > 0\right).
}{d(x)=I(\beta_0 +\beta_1 * x1  + \beta_2 * x2 > 0).}
Polynomial arguments are also supported.}

\item{cluster}{default is FALSE, meaning do not use parallel computing for the genetic algorithm(GA).}

\item{s.tol}{tolerance level for the GA algorithm. This is input for parameter \code{solution.tolerance}
in function \code{rgenoud::genoud}.}

\item{it.num}{the maximum GA iteration number}

\item{pop.size}{an integer with the default set to be 3000. This is roughly the 
number individuals for the first generation
in the genetic algorithm (\code{rgenoud::genoud}).}

\item{Domains}{default is NULL. Otherwise, the object should be a \code{nvars *2} 
matrix used as the space of parameters, which will be supplied to \code{rgenoud::genoud}.}

\item{sign_beta1}{logical. FALSE if the coefficient for the first continuous variable 
is fixed to be negative one; TRUE if positive one.}

\item{Penalty.level}{the level that determines which objective function to use. 
\code{Penalty.level = 0} indicates no regularization;
\code{Penalty.level = 1} indicates the value function estimation minus the means absolute average coefficient 
is the output, which is useful trick to achieve uniqueness of estimated optimal TR
when resolution of input response is low.}
}
\description{
This function supports several user facing functions for Quantile-optimal
treatment regime estimation, namely

 \code{\link{IPWE_Qopt_IndCen}(), \link{IPWE_Qopt_DTR_IndCen}(), \link{IPWE_Qopt_DepCen_trt}(), and \link{IPWE_Qopt_DepCen_general}()}. 
 
It implements the genetic algorithm based policy-search method with 
inverse probability weighting for censored data, such that the estimator is cube root consistent
under the assumption that the propensity score model and the model for the 
survival distriution of the censoring time variable are both correct.
}
\examples{
GenerateData <- function(n)
{
  x1 <- runif(n, min=-0.5,max=0.5)
  x2 <- runif(n, min=-0.5,max=0.5)
  error <- rnorm(n, sd= 1)
  ph <- rep(0.5,n)
  a <- rbinom(n = n, size = 1, prob=ph)
  c <- 1.5 +  + runif(n = n, min=0, max=2)
  cmplt_y <-  pmin(2+x1+x2 +  a*(1 - x1 - x2) +  (0.2 + a*(1+x1+x2)) * error, 4.4)
  censor_y <- pmin(cmplt_y, c)
  delta <- as.numeric(c > cmplt_y)
  return(data.frame(x1=x1,x2=x2,a=a, censor_y = censor_y, delta=delta))
}
n <- 100
data <- GenerateData(n)

# preprocessing
data_aug <- data
data_aug$ph <- rep(mean(data$a), n)
data_aug$deltaC <- 1 - data_aug$delta
library(survival)
survfit_all <- survfit(Surv(censor_y, event = deltaC)~1, data=data_aug)
survest <- stepfun(survfit_all$time, c(1, survfit_all$surv))
data_aug$ghat <- survest(data_aug$censor_y)
data_aug$epsi <- (data_aug$ph * data_aug$a + (1 - data_aug$ph) * (1 - data_aug$a)) * data_aug$ghat

# estimate the median-optimal treatment regime
\donttest{
quantopt_fit <- Gene_Quantile_CenIPWE(data_aug=data_aug,tau=0.5,
                                      p_level=1, regimeClass=a~x1+x2^2, 
                                      sign_beta1=FALSE)
}
\dontshow{
quantopt_fit <- Gene_Quantile_CenIPWE(data_aug=data_aug,tau=0.5, sign_beta1=FALSE,
                                      p_level=1, regimeClass=a~x1+x2,
                                      s.tol=0.4, it.num=1, pop.size=1000)
}
}
