# LindleyRayleigh.R
#' Lindley-Rayleigh Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Lindley-Rayleigh distribution.
#'
#' The Lindley-Rayleigh distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Lindley-Rayleigh distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \theta) =
#' \left[1-e^{-\alpha x^2}\right]^\theta\left\{1-\left(\frac{\theta}
#' {1+\theta}\right) \ln \left(1-e^{-\alpha x^2}\right)\right\} \quad ;\;x > 0.
#' }
#'
#' where\eqn{\alpha} and \eqn{\theta} are the parameters.
#'
#' Included functions are:
#' \itemize{
#'   \item \code{dlindley.rlh()} — Density function
#'   \item \code{plindley.rlh()} — Distribution function
#'   \item \code{qlindley.rlh()} — Quantile function
#'   \item \code{rlindley.rlh()} — Random generation
#'   \item \code{hlindley.rlh()} — Hazard function
#' }
#'
#' @usage
#' dlindley.rlh(x, alpha, theta, log = FALSE)
#' plindley.rlh(q, alpha, theta, lower.tail = TRUE, log.p = FALSE)
#' qlindley.rlh(p, alpha, theta, lower.tail = TRUE, log.p = FALSE)
#' rlindley.rlh(n, alpha, theta)
#' hlindley.rlh(x, alpha, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlindley.rlh}: numeric vector of (log-)densities
#'   \item \code{plindley.rlh}: numeric vector of probabilities
#'   \item \code{qlindley.rlh}: numeric vector of quantiles
#'   \item \code{rlindley.rlh}: numeric vector of random variates
#'   \item \code{hlindley.rlh}: numeric vector of hazard values
#' }
#'
#' @references   
#' Joshi, R. K., & Kumar, V. (2020). 
#' New Lindley-Rayleigh Distribution with Statistical properties and Applications. 
#' \emph{International Journal of Mathematics Trends and Technology (IJMTT)}, 
#' \bold{66(9)}, 197--208.  
#' \doi{10.14445/22315373/IJMTT-V66I9P523}      
#'
#' @examples
#' x <- seq(0.5, 5, 0.25)
#' dlindley.rlh(x, 0.25, 1.5)
#' plindley.rlh(x, 0.25, 1.5)
#' qlindley.rlh(0.75, 0.25, 1.5)
#' rlindley.rlh(10, 0.25, 1.5)
#' hlindley.rlh(x, 0.25, 1.5)
#' 
#' # Data
#' x <- rainfall
#' # ML estimates
#' params = list(alpha=0.2170, theta=1.2107)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plindley.rlh, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlindley.rlh, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlindley.rlh, pfun=plindley.rlh, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LindleyRayleigh
#' @aliases dlindley.rlh plindley.rlh qlindley.rlh rlindley.rlh hlindley.rlh
#' @export
NULL

#' @rdname LindleyRayleigh
#' @usage NULL
#' @export   
dlindley.rlh <- function (x, alpha, theta, log = FALSE)
{
    if (!all(sapply(list(x, alpha, theta), is.numeric))) {
        stop("All arguments must be numeric.")
    }
    if (any(c(alpha, theta) <= 0)) {
        stop("Parameters 'alpha' and 'theta' must be positive.")
    }

    pdf <- numeric(length(x))

    valid <- x > 0
    if (any(valid)) {
	u <- alpha * (x[valid] * x[valid])	
	z <- 1 - exp(-u) 	
    pdf[valid] <- (theta*theta/(1+theta))*2*alpha*x[valid] * exp(-u) * (z ^ (theta-1)) * (1 - log(z))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LindleyRayleigh
#' @usage NULL
#' @export 
plindley.rlh <- function (q, alpha, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, theta) <= 0)) {
		stop("Parameters 'alpha' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- alpha * (q[valid] * q[valid])	
	z <- 1 - exp(-u)
    cdf[valid] <- (z ^ theta) * (1 - (theta/(1+theta)) * log(z))  
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LindleyRayleigh
#' @usage NULL
#' @export 
qlindley.rlh <- function (p, alpha, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, theta) <= 0))
		stop("Parameters 'alpha', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	log((1-exp(-alpha*(x*x)))^-theta)+1+theta-wi * ((1-exp(-alpha*(x*x)))^ -theta) *(1+theta)
	}
	
	for (i in to_solve) {
      wi <- p[i]
      lower <- 1e-3
      upper <- 1e3
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-5),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
}

#' @rdname LindleyRayleigh
#' @usage NULL
#' @export 
rlindley.rlh <- function(n, alpha, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, theta), is.numeric)))
		stop("'alpha', and 'theta' must be numeric.")
	if (any(c(alpha, theta) <= 0))
		stop("Invalid parameters: alpha, and theta must be positive.")

    u <- runif(n)
    q <- qlindley.rlh(u, alpha, theta)
    return(q)
}

#' @rdname LindleyRayleigh
#' @usage NULL
#' @export 
hlindley.rlh <- function (x, alpha, theta)
{
    if (!all(sapply(list(x, alpha, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dlindley.rlh(x, alpha, theta) 
    surv <- 1.0 - plindley.rlh(x, alpha, theta)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 




