# InvPowerCauchy.R
#' Inverse Power Cauchy Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverse Power Cauchy distribution.
#'
#' The Inverse Power Cauchy distribution is parameterized by the parameters
#' \eqn{\alpha > 0} and \eqn{\lambda > 0}.
#'
#' @details
#' The Inverse Power Cauchy distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda) =
#' \quad 1-2 \pi^{-1} \tan ^{-1}\left[\left(\frac{\lambda}{x}\right)
#' ^\alpha\right]   \, ; \quad x > 0.
#' }
#'
#' where \eqn{\alpha} and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dinv.pow.cauchy()} — Density function
#'   \item \code{pinv.pow.cauchy()} — Distribution function
#'   \item \code{qinv.pow.cauchy()} — Quantile function
#'   \item \code{rinv.pow.cauchy()} — Random generation
#'   \item \code{hinv.pow.cauchy()} — Hazard function
#' }
#'
#' @usage
#' dinv.pow.cauchy(x, alpha, lambda, log = FALSE)
#' pinv.pow.cauchy(q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' qinv.pow.cauchy(p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' rinv.pow.cauchy(n, alpha, lambda)
#' hinv.pow.cauchy(x, alpha, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.pow.cauchy}: numeric vector of (log-)densities
#'   \item \code{pinv.pow.cauchy}: numeric vector of probabilities
#'   \item \code{qinv.pow.cauchy}: numeric vector of quantiles
#'   \item \code{rinv.pow.cauchy}: numeric vector of random variates
#'   \item \code{hinv.pow.cauchy}: numeric vector of hazard values
#' }
#'
#' @references 
#' Sapkota L. P., & Kumar V. (2023). 
#' Applications and Some Characteristics of Inverse Power Cauchy Distribution. 
#' \emph{Reliability: Theory & Applications}. \bold{18, 1(72)}, 301--315. 
#' \doi{10.24412/1932-2321-2023-172-301-315}  
#'  
#' Chaudhary, A.K., Sapkota, L.P., & Kumar, V. (2020). 
#' Truncated Cauchy Power–Inverse Exponential distribution: Theory and Applications. 
#' \emph{IOSR Journal of Mathematics (IOSR-JM)}, \bold{16(4)}, Ser.IV, 12--23. 
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dinv.pow.cauchy(x, 2.0, 5.0)
#' pinv.pow.cauchy(x, 2.0, 5.0)
#' qinv.pow.cauchy(0.5, 2.0, 5.0)
#' rinv.pow.cauchy(10, 2.0, 5.0)
#' hinv.pow.cauchy(x, 2.0, 5.0)
#' 
#' # Data
#' x <- headneck44
#' # ML estimates
#' params = list(alpha=1.4271, lambda=123.5294)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.pow.cauchy, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.pow.cauchy, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' res <- gofic(x, params = params,
#'              dfun = dinv.pow.cauchy, pfun=pinv.pow.cauchy, plot=FALSE)
#' print.gofic(res)
#' 
#' @name InvPowerCauchy
#' @aliases dinv.pow.cauchy pinv.pow.cauchy qinv.pow.cauchy rinv.pow.cauchy hinv.pow.cauchy
#' @export
NULL

#' @rdname InvPowerCauchy
#' @usage NULL
#' @export   
dinv.pow.cauchy <- function (x, alpha, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	k <- lambda/x[valid]
	z <- 1.0 + (k^(2*alpha))	
    pdf[valid] <-  ((2/pi) *alpha *(lambda^alpha)) * (x[valid] ^ -(alpha+1)) * (1/z)
	}
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname InvPowerCauchy
#' @usage NULL
#' @export 
pinv.pow.cauchy <- function (q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    cdf[valid] <- 1.0- (2/pi) * atan((lambda/q[valid])^ alpha) 
    }
	if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname InvPowerCauchy
#' @usage NULL
#' @export  
qinv.pow.cauchy <- function (p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- lambda * (tan(0.5 * pi * (1.0 - p[valid])) ^ -(1/alpha))
	}
    return(qtl)   
}

#' @rdname InvPowerCauchy
#' @usage NULL
#' @export 
rinv.pow.cauchy <- function(n, alpha, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")

    u <- runif(n)
    q <- qinv.pow.cauchy(u, alpha, lambda)
    return(q)
}

#' @rdname InvPowerCauchy
#' @usage NULL
#' @export 
hinv.pow.cauchy <- function (x, alpha, lambda)
{
    if (!all(sapply(list(x, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dinv.pow.cauchy(x, alpha, lambda) 
    surv <- 1.0 - pinv.pow.cauchy(x, alpha, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)

    return(hrf)  
} 


