% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MakeTable.R
\name{MakeTable}
\alias{MakeTable}
\title{Create LaTeX Tables From MonteCarlo Output.}
\usage{
MakeTable(output, rows, cols, digits = 4, collapse = NULL,
  transform = NULL, include_meta = TRUE, width_mult = 1,
  partial_grid = NULL)
}
\arguments{
\item{output}{List of class MonteCarlo generated by \code{MonteCarlo}.}

\item{rows}{Vector of parameter names to be stacked in the rows of the table. Ordered from the inside to the outside.}

\item{cols}{Vector of parameter names to be stacked in the cols of the table. Ordered from the inside to the outside.}

\item{digits}{Maximal number of digits displayed in table. Default is \code{digits=4}.}

\item{collapse}{Optional list of the same length as the list returned by the function *func* supplied to \code{MonteCarlo()}.
This list specifies the names of functions to be applied to the repective components of \code{output} when collapsing the results to a table. 
By default means are taken. Another example could be \code{sd()}. Currently, functions supplied have to return a scalar.}

\item{transform}{Optional argument to transform the output table (for example from MSE to RMSE). If a function is supplied
it is applied to all tables. Alternatively, a list of functions can be supplied that has the same length as the list
returned by the function *func* supplied to \code{MonteCarlo()}. 
For tables that are supposed to stay unchanged set list element to \code{NULL}.}

\item{include_meta}{Boolean that determines whether the meta data provided by \code{summary()} is included in comments below the table.
Default is \code{include_meta==TRUE}.}

\item{width_mult}{Scaling factor for width of the output table. Default is \code{width_mult=1}.}

\item{partial_grid}{Optional list with the elements named after the parameters for which only a part of the grid values is
supposed to be included in the table. Each component of the list is a vector that specifies the grid values of interest.}
}
\description{
\code{MakeTable} generates LaTeX tables with user determined ordering from the output of \code{MonteCarlo}.
}
\details{
To generate a two-dimensional table from the high dimensional array of simulation results in output,
the results have to be stacked into rows and columns. The orderning of the resulting table is defined by the ordering
in \code{rows} and \code{cols} that are ordered from the inside of the desired table to the outside.

The first two elements specify a matrix for all possible combinations from the grids for the two desired parameters.
For a third parameter, the matrices for the first two can be stacked in columns 
- one over the other - or in rows - one next to the other. The result of this is a larger matrix. This matrix produced 
for each value of the grid for the fourth parameter can again be stacked into rows or columns and so on. Consult the example.

To compile a Tex document containing the generated table include '\\usepackage\{multirow\}' in the preamble.

To make the resultig tables more comprehensive, parameter grids of length one are dropped from the table 
(unless they are the only value in either cols or rows) and the information is added to the caption. 

In case that the simulation function \code{func} used in \code{MonteCarlo} returns a list with more than one element 
(for example the results of two competing estimators or tests) separate tables are generated for each list element.

If it is desired to include the list elements in a single table, this behavior can be modified by adding "list" 
in one of the vectors \code{rows} or \code{cols} (see examples).
}
\examples{
test_func<-function(n,loc,scale){
 sample<-rnorm(n, loc, scale)
 stat<-sqrt(n)*mean(sample)/sd(sample)
 decision<-abs(stat)>1.96
 return(list("decision"=decision))
}

n_grid<-c(50,100,250,500)
loc_grid<-seq(0,1,0.2)
scale_grid<-c(1,2)

param_list=list("n"=n_grid, "loc"=loc_grid, "scale"=scale_grid)
erg<-MonteCarlo(func=test_func, nrep=250, param_list=param_list, ncpus=1)
str(erg)

rows<-c("n")
cols<-c("loc","scale")
MakeTable(output=erg, rows=rows, cols=cols, digits=2)


#-------- Further Examples: Compare Mean and Median as Estimators for the Expected Value

# define func

func<-function(n,loc,scale){
 
 # generate sample
 sample<-rnorm(n, loc, scale)
 
 # calculate estimators
 mean_sample<-mean(sample)
 median_sample<-median(sample)
 
 # calculate bias
 bias_mean_sample<-mean_sample-loc
 bias_median_sample<-median_sample-loc
 
 # return results
 return(list("mean for calculation of sd"=mean_sample, 
 "bias_mean"=bias_mean_sample, 
 "median for calculation of sd"=median_sample,
 "bias_median"=bias_median_sample))
}

n_grid<-c(50,100,250,500)
loc_grid<-seq(0,1,0.2)
scale_grid<-c(1,2)

param_list=list("n"=n_grid, "loc"=loc_grid, "scale"=scale_grid)
erg_mean_median<-MonteCarlo(func=func, nrep=250, param_list=param_list, ncpus=1)

rows<-c("n")
cols<-c("loc","scale")

# use partial_grid

MakeTable(output=erg_mean_median, rows=rows, cols=cols, digits=2,
          partial_grid=list("n"=c(1,3), "loc"=c(1,3,5)), include_meta=FALSE)

# use collapse to calculate standard deviation and bias

collapse<-list("sd", "mean", "sd", "mean")
MakeTable(output=erg_mean_median, rows=rows, cols=cols, digits=2, 
          collapse=collapse, include_meta=FALSE)

# merge all results in one table

MakeTable(output=erg_mean_median, rows=c("n","loc"), cols=c("scale","list"),
          digits=2, collapse=collapse, include_meta=FALSE)

# transform the results for better scaling

scale_table_10<-function(x){x*10}

MakeTable(output=erg_mean_median, rows=c("n","loc"), cols=c("scale","list"),
          digits=2, collapse=collapse,
          transform=list(scale_table_10, NULL, function(x){x*10}, NULL),
          include_meta=FALSE)

}
