% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{as.Mclust}
\alias{as.Mclust}
\alias{as.Mclust.MoEClust}
\title{Convert MoEClust objects to the Mclust class}
\usage{
\method{as.Mclust}{MoEClust}(x,
         expert.covar = TRUE,
         signif = 0L,
         ...)
}
\arguments{
\item{x}{An object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}} or an object of class \code{"MoECompare"} generated by \code{\link{MoE_compare}}. Models with a noise component are facilitated here too.}

\item{expert.covar}{Logical (defaults to \code{TRUE}) governing whether the extra variability in the component means is added to the MVN ellipses corresponding to the component covariance matrices in the presence of expert network covariates. See the function \code{\link{expert_covar}}.}

\item{signif}{Significance level for outlier removal. Must be a single number in the interval [0, 1). Corresponds to the percentage of data to be considered extreme and therefore removed (half of \code{signif} at each endpoint, on a column-wise basis). The default, \code{0}, corresponds to no outlier removal. \strong{Only} invoke this argument as an aid to visualisation via \code{\link[mclust]{plot.Mclust}}.}

\item{...}{Further arguments to be passed to other methods.}
}
\value{
An object of class \code{"Mclust"}. See \code{methods(class="Mclust")} for a (non-exhaustive) list of functions which can be applied to this class.
}
\description{
Converts an object of class \code{"MoEClust"} generated by \code{\link{MoE_clust}} and converts it to an object of class \code{"Mclust"} as generated by fitting \code{\link[mclust]{Mclust}}, to facilitate use of plotting and other functions for the \code{"Mclust"} class within the \pkg{mclust} package. Some caution is advised when converting models with gating &/or expert covariates (see Note below) and users are always encouraged to use the dedicated \code{\link{plot.MoEClust}} function for objects of the \code{"MoEClust"} class instead.
}
\details{
Mixing proportions are averaged over observations in components in the presence of gating network covariates during the coercion. For models with expert network covariates, the means are given by the posterior means of the fitted values of the expert network.

In the presence of expert network covariates, the component-specific covariance matrices are (by default, via the argument \code{expert.covar}) modified for plotting purposes via the function \code{\link{expert_covar}}, in order to account for the extra variability of the means, usually resulting in bigger shapes & sizes for the MVN ellipses.

The \code{signif} argument is intended only to aid visualisation via \code{\link[mclust]{plot.Mclust}}, as plots therein can be sensitive to outliers, particularly with regard to axis limits. However, users are always encouraged to use the dedicated \code{\link{plot.MoEClust}} function for objects of the \code{"MoEClust"} class instead (see Note below).
}
\note{
Plots may be quite misleading in the presence of gating &/or (especially) expert network covariates when the \code{what} argument is \code{"density"} within \code{\link[mclust]{plot.Mclust}}; users are \strong{strongly} encouraged to use \code{\link{MoE_gpairs}} with \code{response.type="density"} &/or \code{show.dens=TRUE} instead.

Predictions (via \code{\link[mclust]{predict.Mclust}}) will also be misleading in the presence of covariates of any kind when \code{newdata} is supplied; thus, users are \strong{strongly} encouraged to use \code{\link{predict.MoEClust}} instead. 

The functions \code{\link[mclust]{clustCombi}} and \code{\link[mclust]{clustCombiOptim}} can be safely used (provided \code{as.Mclust(x)} is supplied as the \code{object} argument to \code{\link[mclust]{clustCombi}}), as they only rely on \code{x$z} and \code{x$G} only. See the examples below.

Users may expect MoEClust models with no covariates of any kind to be identical to models fitted via \pkg{mclust}, but this is not necessarily true: see the \code{\link{MoE_control}} argument \code{asMclust}.
}
\examples{
\donttest{library(mclust)

# Fit a gating network mixture of experts model to the ais data
data(ais)
mod   <- MoE_clust(ais[,3:7], G=3, gating= ~ BMI + sex, modelNames="EEE", network.data=ais)

# Convert to the "Mclust" class and examine the classification
mod2  <- as.Mclust(mod)
plot(mod2, what="classification")

# Examine the uncertainty
plot(mod2, what="uncertainty")

# Return the optimal number of clusters according to entropy
combi <- mclust::clustCombi(object=mod2)
optim <- mclust::clustCombiOptim(object=combi)
table(mod2$classification, ais$sex)
table(optim$cluster.combi, ais$sex)

# Compare plot.MoEClust and plot.Mclust for univariate mixtures
data(CO2data)
res <- MoE_clust(CO2data$CO2, G=2, expert = ~ GNP, modelNames="V", network.data=CO2data)
plot(as.Mclust(res), what="classification")
plot(as.Mclust(res), what="density")
plot(as.Mclust(res), what="uncertainty") 

# Proper version of what="density" plot:
MoE_gpairs(res, show.map=FALSE, cov.ind=0, show.dens=TRUE)

# Equivalent what="uncertainty" plot:
MoE_Uncertainty(res)}
}
\references{
Fraley, C. and Raftery, A. E. (2002). Model-based clustering, discriminant analysis, and density estimation. \emph{Journal of the American Statistical Association}, 97(458): 611-631.

Scrucca L., Fop M., Murphy T. B. and Raftery A. E. (2016). mclust 5: clustering, classification and density estimation using Gaussian finite mixture models. \emph{The R Journal}, 8(1): 289-317.
}
\seealso{
\code{\link[mclust]{Mclust}}, \code{\link[mclust]{plot.Mclust}}, \code{\link{MoE_clust}}, \code{\link{plot.MoEClust}}, \code{\link{predict.MoEClust}}, \code{\link{expert_covar}}, \code{\link{MoE_control}}, \code{\link[mclust]{clustCombi}}, \code{\link[mclust]{clustCombiOptim}}
}
\author{
Keefe Murphy - <\email{keefe.murphy@mu.ie}>
}
\keyword{utility}
