# =============================================================================
# RUN ALL EM ESTIMATIONS WITHOUT GIBBS
# =============================================================================

#' Run all EM-based estimations without Gibbs sampling (CRAN-safe)
#'
#' Executes multiple EM algorithms (recursive ECF, kernel ECF, weighted OLS, CDF-based)
#' on the input data. Optionally saves mixture plots to a temporary directory.
#'
#' @param X1 Numeric vector of data.
#' @param bw_sj Numeric. Bandwidth value (e.g., Silverman's rule).
#' @param max_iter Integer. Maximum number of EM iterations.
#' @param tol Numeric. Convergence tolerance.
#' @param save_plots Logical. Whether to save PNG plots to tempdir().
#' @return Invisibly returns a list containing fitted parameters for each EM method.
#' @export
run_all_estimations <- function(X1, bw_sj, max_iter = 200, tol = 1e-4, save_plots = FALSE) {
  X1 <- X1[!is.na(X1)]
  message("Bande passante R (bw.SJ): ", sprintf("%.5f", bw_sj))

  # Helper to run EM with error handling
  safe_em <- function(f) {
    tryCatch(f(X1, max_iter, tol),
             error = function(e) {
               message("EM estimation failed: ", e$message)
               list(
                 params1 = list(alpha = NA, beta = NA, gamma = NA, delta = NA),
                 params2 = list(alpha = NA, beta = NA, gamma = NA, delta = NA),
                 weight = 0.5
               )
             })
  }

  # Run all EM estimations
  result1   <- safe_em(em_estimate_stable_recursive_ecf)
  result11  <- safe_em(em_estimate_stable_kernel_ecf)
  result111 <- safe_em(em_estimate_stable_weighted_ols)
  result1111<- safe_em(em_estimate_stable_from_cdf)

  # Plotting function wrapper
  plot_safe <- function(res, filename) {
    if (save_plots) {
      plot_mixture_fit(X1, list(
        weights = c(res$weight, 1 - res$weight),
        alphas  = c(res$params1$alpha, res$params2$alpha),
        betas   = c(res$params1$beta, res$params2$beta),
        gammas  = c(res$params1$gamma, res$params2$gamma),
        deltas  = c(res$params1$delta, res$params2$delta)
      ), save_path = filename, show_plot = FALSE)
      message("Saved plot to tempdir(): ", file.path(tempdir(), filename))
    }
  }

  plot_safe(result1, "recursive_ecf.png")
  plot_safe(result11, "kernel_ecf.png")
  plot_safe(result111, "weighted_ols.png")
  plot_safe(result1111, "cdf_based.png")

  invisible(list(
    recursive_ecf = result1,
    kernel_ecf    = result11,
    weighted_ols  = result111,
    cdf_based     = result1111
  ))
}

# =============================================================================
# RUN ALL EM ESTIMATIONS WITH GIBBS SAMPLING
# =============================================================================

#' Run all EM-based estimations with Gibbs sampling (CRAN-safe)
#'
#' Executes EM algorithms with Gibbs sampling on the input data.
#' Optionally saves mixture plots to a temporary directory.
#'
#' @param data Numeric vector of observations.
#' @param bw_sj Numeric. Bandwidth value (e.g., Silverman's rule).
#' @param max_iter Integer. Maximum number of EM iterations.
#' @param tol Numeric. Convergence tolerance.
#' @param save_plots Logical. Whether to save PNG plots to tempdir().
#' @return Invisibly returns a list containing fitted parameters for each EM method with Gibbs.
#' @export
run_estimations_with_gibbs <- function(data, bw_sj, max_iter = 100, tol = 1e-4, save_plots = FALSE) {
  X1 <- as.numeric(data)
  X1 <- X1[!is.na(X1)]
  message("Bande passante R (bw.SJ): ", sprintf("%.5f", bw_sj))

  safe_em <- function(f) {
    tryCatch(f(X1, max_iter, tol),
             error = function(e) {
               message("EM estimation with Gibbs failed: ", e$message)
               list(
                 params1 = list(alpha = NA, beta = NA, gamma = NA, delta = NA),
                 params2 = list(alpha = NA, beta = NA, gamma = NA, delta = NA),
                 weight = 0.5
               )
             })
  }

  result1   <- safe_em(em_estimate_stable_recursive_ecf_with_gibbs)
  result11  <- safe_em(em_estimate_stable_kernel_ecf_with_gibbs)
  result111 <- safe_em(em_estimate_stable_weighted_ols_with_gibbs)
  result1111<- safe_em(em_estimate_stable_from_cdf_with_gibbs)

  plot_safe <- function(res, filename) {
    if (save_plots) {
      plot_mixture_fit(X1, list(
        weights = c(res$weight, 1 - res$weight),
        alphas  = c(res$params1$alpha, res$params2$alpha),
        betas   = c(res$params1$beta, res$params2$beta),
        gammas  = c(res$params1$gamma, res$params2$gamma),
        deltas  = c(res$params1$delta, res$params2$delta)
      ), save_path = filename, show_plot = FALSE)
      message("Saved plot to tempdir(): ", file.path(tempdir(), filename))
    }
  }

  plot_safe(result1, "recursive_ecf_gibbs.png")
  plot_safe(result11, "kernel_ecf_gibbs.png")
  plot_safe(result111, "weighted_ols_gibbs.png")
  plot_safe(result1111, "cdf_based_gibbs.png")

  invisible(list(
    recursive_ecf = result1,
    kernel_ecf    = result11,
    weighted_ols  = result111,
    cdf_based     = result1111
  ))
}
