# Gibbs, Metropolis-Hastings, and mock samplers for stable mixtures
# =============================================================================
# MOCK GIBBS SAMPLING
# =============================================================================


#' Mock Gibbs sampling for alpha-stable mixture estimation
#'
#' Performs a simplified Gibbs sampling procedure to estimate parameters of a two-component
#' alpha-stable mixture. Samples are drawn from prior distributions and evaluated using
#' log-likelihood. The best parameter set is selected based on likelihood.
#'
#' @param data Numeric vector of observations.
#' @param n_samples Number of Gibbs samples to draw.
#' @param verbose Logical, whether to print best log-likelihood.
#' @return List containing best_params and all sampled parameter sets.
#' @importFrom stats rnorm runif rbeta sd
#' @export
mock_gibbs_sampling <- function(data, n_samples = 500, verbose = FALSE) {
  samples <- list()
  best_loglik <- -Inf
  best_params <- NULL

  data_mean <- mean(data)
  data_std  <- sd(data)

  for (i in 1:n_samples) {
    tryCatch({
      # Component 1 priors (heavier tail)
      alpha1 <- runif(1, 0.8, 1.6)
      beta1  <- runif(1, -1.0, 1.0)
      gamma1 <- max(abs(rnorm(1, 1.0, 0.5)), 0.1)
      delta1 <- rnorm(1, data_mean, data_std)

      # Component 2 priors (lighter tail)
      alpha2 <- runif(1, 1.2, 2.0)
      beta2  <- runif(1, -1.0, 1.0)
      gamma2 <- max(abs(rnorm(1, 1.5, 0.5)), 0.1)
      delta2 <- rnorm(1, data_mean + 2.0, data_std)

      # Mixture weight prior
      w <- pmax(pmin(rbeta(1, 2, 2), 0.95), 0.05)

      candidate <- c(w, alpha1, beta1, gamma1, delta1, alpha2, beta2, gamma2, delta2)
      ll <- -log_likelihood_mixture(candidate, data)

      samples[[i]] <- list(ll = ll, params = candidate)

      if (ll > best_loglik) {
        best_loglik <- ll
        best_params <- candidate
      }
    }, error = function(e) {
      # Skip invalid parameter sets silently
    })
  }

  if (verbose) {
    message("Best Gibbs+MLE Log-Likelihood:", round(best_loglik, 2), "\n")
  }

  return(list(best_params = best_params, samples = samples))
}

# =============================================================================
# GIBBS SAMPLER FOR GAUSSIAN MIXTURE
# =============================================================================


#' Gibbs sampler for Gaussian mixture model
#'
#' Performs Gibbs sampling for a two-component Gaussian mixture model.
#' Updates latent assignments, component means, variances, and mixing proportions.
#'
#' @param data Numeric vector of observations.
#' @param iterations Number of Gibbs iterations.
#' @return List of sampled parameters per iteration.
#' @importFrom stats dnorm sd
#' @export
gibbs_sampler <- function(data, iterations = 1000) {
  if (length(data) < 2) {
    stop("Data must contain at least two points.")
  }

  n <- length(data)
  z <- sample(c(0, 1), n, replace = TRUE)
  mu <- c(mean(data) - 1, mean(data) + 1)
  sigma <- c(1.0, 1.0)
  pi_val <- 0.5

  samples <- list()

  for (it in 1:iterations) {
    # Step 1: Sample z
    for (i in 1:n) {
      p0 <- pi_val * dnorm(data[i], mu[1], sigma[1])
      p1 <- (1 - pi_val) * dnorm(data[i], mu[2], sigma[2])
      total_prob <- p0 + p1
      z[i] <- if (total_prob > 0) sample(c(0, 1), 1, prob = c(p0 / total_prob, p1 / total_prob)) else sample(c(0, 1), 1)
    }

    # Step 2: Sample mu and sigma
    for (j in 0:1) {
      group <- data[z == j]
      if (length(group) > 0) {
        mu[j + 1] <- mean(group)
        sigma[j + 1] <- sd(group)
      }
    }

    # Step 3: Sample pi
    pi_val <- mean(z == 0)

    samples[[it]] <- list(mu1 = mu[1], mu2 = mu[2],
                          sigma1 = sigma[1], sigma2 = sigma[2],
                          pi = pi_val)
  }

  return(samples)
}

# =============================================================================
# METROPOLIS-HASTINGS SAMPLER
# =============================================================================


#' Metropolis-Hastings MCMC for stable mixture clustering
#'
#' Performs Metropolis-Hastings sampling to estimate parameters of a two-component
#' alpha-stable mixture model. Proposals are generated for each parameter and accepted
#' based on likelihood ratios. Cluster assignments are updated at each iteration.
#'
#' @param fct Function to estimate initial parameters.
#' @param iterations Number of MCMC iterations.
#' @param lok Numeric vector of observations.
#' @param aa Prior parameters for Dirichlet distribution.
#' @param proposal_std Standard deviation for proposal distributions.
#' @return List of sampled parameters and weights across iterations.
#' @importFrom stats rnorm rgamma rbinom kmeans
#' @export
metropolis_hastings <- function(fct, iterations, lok, aa = c(1, 1), proposal_std = 0.1) {
  n <- length(lok)
  u1 <- seq(0.1, 1.0, length.out = 10)

  # Initial clustering
  km_result <- kmeans(matrix(lok, ncol = 1), centers = 2)
  a <- km_result$cluster - 1  # 0/1

  lok1 <- fct(lok[a == 0], u1)
  lok2 <- fct(lok[a == 1], u1)

  # Initialize parameters...
  M2_alpha1 <- c(lok1$alpha); M2_beta1  <- c(lok1$beta)
  M2_delta1 <- c(lok1$delta); M2_omega1 <- ensure_positive_scale(c(lok1$gamma))
  M2_alpha2 <- c(lok2$alpha); M2_beta2  <- c(lok2$beta)
  M2_delta2 <- c(lok2$delta); M2_omega2 <- ensure_positive_scale(c(lok2$gamma))

  M2_w1 <- c(mean(a == 0)); M2_w2 <- c(mean(a == 1))
  M2_cc <- matrix(0, nrow = iterations + 1, ncol = n)
  M2_cc[1, ] <- a + 1

  # Helpers
  propose_unbounded <- function(param, sd) rnorm(1, param, sd)
  # log-normal random-walk to keep scale > 0
  propose_positive <- function(param, sd) exp(rnorm(1, log(param), sd))

  safe_logpdf_sum <- function(x, alpha, beta, scale, location) {
    if (length(x) == 0) return(0)
    sum(log(pmax(r_stable_pdf(x, alpha, beta, scale, location), 1e-12)))
  }

  for (s in 1:iterations) {
    cc <- M2_cc[s, ]

    # Update weights
    counts <- c(sum(cc == 1), sum(cc == 2))
    w <- rgamma(2, aa + counts)
    w <- w / sum(w)
    w1 <- w[1]; w2 <- w[2]
    M2_w1 <- c(M2_w1, w1); M2_w2 <- c(M2_w2, w2)

    # --- Proposals (alpha in (0,2], beta in [-1,1], scale > 0, location any real) ---
    alpha1_star <- min(2, max(0.1, propose_unbounded(M2_alpha1[s], proposal_std)))
    beta1_star  <- max(-1, min(1,  propose_unbounded(M2_beta1[s],  proposal_std)))
    delta1_star <- propose_unbounded(M2_delta1[s], proposal_std)
    omega1_star <- ensure_positive_scale(propose_positive (M2_omega1[s], proposal_std)  )

    alpha2_star <- min(2, max(0.1, propose_unbounded(M2_alpha2[s], proposal_std)))
    beta2_star  <- max(-1, min(1,  propose_unbounded(M2_beta2[s],  proposal_std)))
    delta2_star <- propose_unbounded(M2_delta2[s], proposal_std)
    omega2_star <- ensure_positive_scale(propose_positive (M2_omega2[s], proposal_std) )

    # Pack as (alpha, beta, scale, location)  <-- IMPORTANT
    p1_star <- c(alpha1_star, beta1_star, omega1_star, delta1_star)
    p2_star <- c(alpha2_star, beta2_star, omega2_star, delta2_star)

    p1 <- c(M2_alpha1[s], M2_beta1[s], M2_omega1[s], M2_delta1[s])
    p2 <- c(M2_alpha2[s], M2_beta2[s], M2_omega2[s], M2_delta2[s])

    # Log densities by cluster (robust to empty)
    idx1 <- cc == 1; idx2 <- cc == 2
    ll1_cur <- safe_logpdf_sum(lok[idx1], p1[1], p1[2], p1[3], p1[4])
    ll1_pro <- safe_logpdf_sum(lok[idx1], p1_star[1], p1_star[2], p1_star[3], p1_star[4])
    ll2_cur <- safe_logpdf_sum(lok[idx2], p2[1], p2[2], p2[3], p2[4])
    ll2_pro <- safe_logpdf_sum(lok[idx2], p2_star[1], p2_star[2], p2_star[3], p2_star[4])

    # Accept/reject
    accept1 <- min(1, exp(ll1_pro - ll1_cur))
    accept2 <- min(1, exp(ll2_pro - ll2_cur))

    if (runif(1) < accept1) {
      M2_alpha1 <- c(M2_alpha1, p1_star[1]); M2_beta1  <- c(M2_beta1,  p1_star[2])
      M2_omega1 <- c(M2_omega1, p1_star[3]); M2_delta1 <- c(M2_delta1, p1_star[4])
    } else {
      M2_alpha1 <- c(M2_alpha1, p1[1]);      M2_beta1  <- c(M2_beta1,  p1[2])
      M2_omega1 <- c(M2_omega1, p1[3]);      M2_delta1 <- c(M2_delta1, p1[4])
    }

    if (runif(1) < accept2) {
      M2_alpha2 <- c(M2_alpha2, p2_star[1]); M2_beta2  <- c(M2_beta2,  p2_star[2])
      M2_omega2 <- c(M2_omega2, p2_star[3]); M2_delta2 <- c(M2_delta2, p2_star[4])
    } else {
      M2_alpha2 <- c(M2_alpha2, p2[1]);      M2_beta2  <- c(M2_beta2,  p2[2])
      M2_omega2 <- c(M2_omega2, p2[3]);      M2_delta2 <- c(M2_delta2, p2[4])
    }

    # Update cluster assignments using latest params
    params1 <- c(tail(M2_alpha1, 1), tail(M2_beta1, 1), tail(M2_omega1, 1), tail(M2_delta1, 1))
    params2 <- c(tail(M2_alpha2, 1), tail(M2_beta2, 1), tail(M2_omega2, 1), tail(M2_delta2, 1))

    log_pdf1 <- log(pmax(r_stable_pdf(lok, params1[1], params1[2], params1[3], params1[4]), 1e-12))
    log_pdf2 <- log(pmax(r_stable_pdf(lok, params2[1], params2[2], params2[3], params2[4]), 1e-12))

    log_prob1 <- log(pmax(w1, 1e-12)) + log_pdf1
    log_prob2 <- log(pmax(w2, 1e-12)) + log_pdf2
    max_log <- pmax(log_prob1, log_prob2)
    prob1 <- exp(log_prob1 - max_log)
    prob2 <- exp(log_prob2 - max_log)
    prob1 <- prob1 / (prob1 + prob2)

    # draw new cluster labels (1 or 2)
    new_cc <- rbinom(n, 1, 1 - prob1) + 1
    M2_cc[s + 1, ] <- new_cc

    message("Iteration", s, "/", iterations, "completed.\n")
  }

  return(list(
    M2_w1 = M2_w1, M2_w2 = M2_w2,
    M2_alpha1 = M2_alpha1, M2_beta1 = M2_beta1,
    M2_delta1 = M2_delta1, M2_omega1 = M2_omega1,
    M2_alpha2 = M2_alpha2, M2_beta2 = M2_beta2,
    M2_delta2 = M2_delta2, M2_omega2 = M2_omega2
  ))


  }

