
<!-- README.md is generated from README.Rmd. Please edit that file -->

# MaximinInfer

<!-- badges: start -->

[![CRAN
status](https://www.r-pkg.org/badges/version/MaximinInfer)](https://CRAN.R-project.org/package=MaximinInfer)
<!-- badges: end -->

MaximinInfer is a package that implements the sampling and aggregation
method for the covariate shift maximin effect, which was proposed in
\<arXiv:2011.07568\>. It constructs the confidence interval for any
linear combination of the high-dimensional maximin effect.

## Installation

You can install the released version of MaximinInfer from
[CRAN](https://CRAN.R-project.org) with:

``` r
install.packages("MaximinInfer")
```

And the development version from [GitHub](https://github.com/) with:

``` r
# install.packages("devtools")
devtools::install_github("zywang0701/MaximinInfer")
```

## Example

This is a basic example which shows you how to solve a common problem:

``` r
library(MaximinInfer)
```

The data is heterogeneous and covariates shift between source and target
data

``` r
## number of groups
L=2
## dimension
p=100

## mean vector for source
mean.source = rep(0, p)
## covariance matrix for source
A1gen <- function(rho,p){
  A1=matrix(0,p,p)
  for(i in 1:p){
    for(j in 1:p){
      A1[i,j]<-rho^(abs(i-j))
    }
  }
  return(A1)
}
cov.source = A1gen(0.6, p)

## 1st group's source data
n1 = 100
X1 = MASS::mvrnorm(n1, mu=mean.source, Sigma=cov.source)
# true coef for 1st group
b1 = rep(0, p)
b1[1:5] = seq(1,5)/20
b1[98:100] = c(0.5, -0.5, -0.5)
Y1 = X1%*%b1 + rnorm(n1)

## 2nd group's source data
n2 = 100
X2 = MASS::mvrnorm(n2, mu=mean.source, Sigma=cov.source)
# true coef for 2nd group
b2 = rep(0, p)
b2[6:10] = seq(1,5)/20
b2[98:100] = 0.5*c(0.5, -0.5, -0.5)
Y2 = X2%*%b2 + rnorm(n2)

## Target Data, covariate shift
n0 = 100
mean0 = rep(0, p)
cov0 = cov.source
for(i in 1:p) cov0[i, i] = 1.5
for(i in 1:5) for(j in 1:5) if(i!=j) cov0[i, j] = 0.9
for(i in 99:100) for(j in 99:100) if(i!=j) cov0[i, j] = 0.9
X0 = MASS::mvrnorm(n0, mu=mean0, Sigma=cov0)
```

Input the loading. Note that it allows for multiple loading
simultaneously.

``` r
loading.mat = matrix(0, nrow=100, ncol=2) # dimension p=100
loading.mat[96:100, 1] = 0.4
loading.mat[99:100, 2] = 0.8
```

Call function `Maximin()`. By setting the argument verbose, you can
choose whether or not to display the intermediate bias-correction
results.

``` r
mm <- Maximin(list(X1,X2), list(Y1,Y2), loading.mat, X0, cov.shift=TRUE, verbose=TRUE)
#> ======> Bias Correction for initial estimators.... 
#> Computing LF for loading (1/2)... 
#> The projection direction is identified at mu = 0.026739at step =6
#> Computing LF for loading (2/2)... 
#> The projection direction is identified at mu = 0.040108at step =5
#> Computing LF for loading (1/2)... 
#> The projection direction is identified at mu = 0.026739at step =6
#> Computing LF for loading (2/2)... 
#> The projection direction is identified at mu = 0.026739at step =6
#> ======> Bias Correction for matrix Gamma.... 
#> Computing LF for loading (1/1)... 
#> The projection direction is identified at mu = 0.026739at step =6
#> Computing LF for loading (1/1)... 
#> The projection direction is identified at mu = 0.026739at step =6
#> Computing LF for loading (1/1)... 
#> The projection direction is identified at mu = 0.005282at step =10
#> Computing LF for loading (1/1)... 
#> The projection direction is identified at mu = 0.007923at step =9
```

The following inference method is:

``` r
out <- Infer(mm, gen.size=200)
```

The weights for each group:

``` r
out$weight
#> [1] 0.5703927 0.4296073
```

The point estimator and its corresponding confidence interval for each
loading:

``` r
out$mminfer
#> [[1]]
#> [[1]]$point
#> [1] -0.212938
#> 
#> [[1]]$CI
#>           lower      upper
#> [1,] -0.4136389 0.01993818
#> 
#> 
#> [[2]]
#> [[2]]$point
#> [1] -0.6861211
#> 
#> [[2]]$CI
#>         lower      upper
#> [1,] -1.20779 -0.1704235
```
