\name{normn MA}
\alias{normn_MA}
\title{Multi-dimensional MA normalization for plate effect}
\description{
Normalize data to minimize the difference among the subgroups of the samples generated by experimental factor such as multiple plates (batch effects)\cr
 - the primary method is Multi-MA, but other fitting function, \emph{f} in manuscript (e.g. loess) is available, too.\cr
 This method is based on the assumptions stated below\cr
 \enumerate{
 \item The geometric mean value of the samples in each subgroup (or plate) for a single target is ideally same as those from the other subgroups.
 \item The subgroup (or plate) effects that influence those mean values for multiple observed targets are dependent on the values themselves. (intensity dependent effects)
 }
}
\usage{
normn_MA(mD, expGroup, represent_FUN= function(x) mean(x, na.rm= T), 
        fitting_FUN= NULL, isLog= TRUE) 
}
\arguments{
	\item{mD}{ a \code{matrix} of measured values in which columns are the measured molecules and rows are samples }
	\item{expGroup}{ a \code{vector} of experimental grouping variable such as plate. The length of code{expGroup} must be same as the number of rows of \code{mD}.}
	\item{represent_FUN}{ a \code{function} that computes representative values for each experimental group (e.g. plate). The default is mean ignoring any NA }
	\item{fitting_FUN}{ \code{NULL} or a \code{function} that fits to data in MA-coordinates. 
	If it is \code{NULL} as the default, 'Multi-MA' method is employed. 
	If a \code{function} is used, two arguments of \code{m_j} and \code{A} are required, which are \eqn{\mathbf{m}_j} coordinate in \eqn{M_d} and \eqn{A} coordinate, respectively. }
	\item{isLog}{ TRUE or FALSE, if the normalization should be conducted after log-transformation. The affinity proteomics data from suspension bead arrays is recommended to be normalized using the default, \code{isLog = TRUE}.}
}
\value{
 The data after normalization in a \code{matrix}
}
\references{
	Hong M-G, Lee W, Pawitan Y, Schwenk JM (201?) 
	Multi-dimensional normalization of plate effects for multiplexed applications
	\emph{unpublished}
}
\author{Mun-Gwan Hong <\email{mun-gwan.hong@scilifelab.se}>}
\examples{
data(sba)
B <- normn_MA(sba$X, sba$plate)		# Multi-MA normalization

# MA-loess normalization
B <- normn_MA(sba$X, sba$plate, fitting_FUN= function(m_j, A) loess(m_j ~ A)$fitted)

# weighted linear regression normalization
B <- normn_MA(sba$X, sba$plate, fitting_FUN= function(m_j, A) {
	beta <- lm(m_j ~ A, weights= 1/A)$coefficients
	beta[1] + beta[2] * A
})

# robust linear regression normalization
if(any(search() == "package:MASS")) {	# excutable only when MASS package was loaded.
	B <- normn_MA(sba$X, sba$plate, fitting_FUN= function(m_j, A) {
		beta <- rlm(m_j ~ A, maxit= 100)$coefficients
		beta[1] + beta[2] * A
	})
}
}

