#'@title Change-point tests for generalized Ornstein-Uhlenbec (GOU) process
#'
#'@description Function to simulate exact N+K+1 values with change point after N+K_star, with  K_star = floor(N*t_star), for a GOU process. Starting point is 0.
#'
#'@param X     observations
#'@param T1    last time of observation
#'@param N     number of observations on from on interval (0,T1]
#'@param p     number of cosine coefficients >=1
#'@param q     number of sine coefficients >=0
#'@param gamma weight parameter >=0 and < 0.5
#'@param c1    critical value for Q stat (based on 1-dimensional weigthed BM)
#'@param cd    critical value for G stat (based on d-dimensional weigthed BM), where d = p+q+1 is the number of estimated parameters for the drift.
#'@return \item{out}{List }
#'
#'@examples
#' T1=20
#' N=500
#' gamma = 0.1
#' p=2
#' q=0
#' c1 = 2.2838 # corresponding to gamma=0.1
#' c3 = 3.0502 # corresponding to gamma=0.1 and d=3 estimated parameters for the drift
#' data(X)
#' out=StatGOU(X,T1,N,p,q,gamma,c1,c3)
#' @references Lyu, Nasri and Remillard (2025): Sequential Change-point Detection with Generalized Ornstein–Uhlenbeck Processes
#'
#'@export

StatGOU=function(X,T1,N,p,q,gamma,c1,cd)
{

  DeltaN = T1/N # time increment Delta t

  n1=length(X)
  n=n1-1
  Y =matrix( diff(X),ncol=1)/ sqrt(DeltaN)
  tt = (c(1:n)-1)* DeltaN


  phi=matrix(1,ncol=1,nrow=n)
  if(p>1)
    {
     for(k in 1:(p-1))
      {
        phi=cbind(phi,fcos(tt*k))
      }
    }
  psi=NULL
    if(q>0)
      {
      for(k in 1:q)
       {
         psi=cbind(psi,fsin(tt*k))
       }
      }
  base=cbind(phi,psi)

  Z = cbind(base,-X[-n1])*sqrt(DeltaN)

  YN = Y[1:N]
  ZN = Z[1:N,]
  S= t(ZN) %*% ZN /T1
  sigmaHat = sqrt(sum(YN^2))/sqrt(N)
  thetaN = coef(lm(YN ~0+ZN))
  res = Y-Z%*%thetaN
  resN = res[(N+1):n]
  Q = abs(cumsum(resN))/sigmaHat
  ttt=c((N+1):n)/N -1
  g1 = sqrt(N)*(1+ttt)*((1-1/(1+ttt))^gamma)
  statQ=Q/g1
  ind=which(statQ>c1)
  tauQ = min(ind,(n1-N))/N


  for (k in seq(from = N + 1, to = n, by =1)) {

    Yk <- Y[1:k]
    Zk <- Z[1:k, ]
    coef_k <- coef(lm(Yk ~ 0+Zk))

    coef_diff <- thetaN - coef_k # Difference between two coefs estimators
    hN <-  ((k - N) /k) ^ gamma

    statG = sqrt(T1)*sqrt( t(coef_diff)%*%S%*%coef_diff)/hN
    if (statG > cd) {
      pos <- (k - N)/(N)
      break
    } else   pos <- n1/N -1

  }
  tauG=pos

  noms=rep(NA,(p+q+1))
  for(j in 1:(p+q))
  {
    noms[j]=paste("mu",j,sep="")
  }
  noms[p+q+1]="alpha"

  names(thetaN)=noms

  timeG = N*(1+tauG)
  timeQ = N*(1+tauQ)

  out=list(timeG=timeG,timeQ=timeQ,tauG=tauG,tauQ=tauQ,sigma.est=sigmaHat,theta.est=thetaN,Sigma.hat=S)
  return(out)
}



