% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comp.var.r
\name{comp.var}
\alias{comp.var}
\title{Components of variation}
\usage{
comp.var(object, anova.tab = NULL)
}
\arguments{
\item{object}{an object of class "\code{\link{lm}}", containing the specified design with random and/or fixed factors.}

\item{anova.tab}{an object containing the results returned by \code{gad} or \code{pooling} functions. See examples below.}
}
\value{
{
A list of length 2, containing the mean squares estimates (\code{$mse}) and the table of components of variation (\code{$comp.var}) for the model. \cr

The \code{$comp.var} table contains four columns: “Type” shows whether terms are fixed or random; “Estimate” shows the estimate of component of variation
(in squared units); “Square root” shows the square root of the “Estimate” column (original unit); and “Percentage” shows the percentage of variability
attributable to each term (both fixed and random factors).
}
}
\description{
This function calculates components of variation for fixed and random factors.
}
\details{
{
This function calculates components of variation for any combination of orthogonal/nested and fixed/random factors. For pooled terms,
\code{comp.var} function seeks for the denominator of the removed term and keep its type (fixed or random). \cr

Note that there are differences on the interpretation of the results between fixed and random factors. For fixed factors, the components of
variation calculated are just the sum of squares divided by the degrees of freedom and the hypothesis only concern those levels that were
included in the model. On the other hand, for random factors the components of variation calculated are truly variance components and are
interpretable as measures of variability of a population of levels, which are randomly sampled. However, for most of studies that aims to
compare the amount of variation attributed to each term, the estimates of components of variation for both fixed and random factors are important
and directly comparable (Anderson et al., 2008). \cr

Eventually, the estimates of components of variation for some terms in the model can be negative. The term in question generally has a large p-value
and its estimative is usually set to zero, since a negative estimate is illogical. An alternative to this problem is remove the term using
\code{\link{pooling}} function and re-analyse the model (Fletcher and Underwood, 2002).
}
}
\examples{
library(GAD)
data(crabs)
head(crabs)

Re <- as.random(crabs$Region)   # random factor
Lo <- as.random(crabs$Location) # random factor nested in Region
Si <- as.random(crabs$Site)     # random factor nested in Location

model <- lm(Density ~ Re + Lo\%in\%Re + Si\%in\%Lo\%in\%Re, data = crabs)

C.test(model)                     # Checking homogeneity of variances
estimates(model)                  # Checking model structure
model.tab <- gad(model)           # Storing the result of ANOVA on a new object
model.tab                         # Checking ANOVA results
comp.var(model, anova.tab = model.tab) # Calculating the components of variations
}
\references{
Anderson, M.J., Gorley, R.N., Clarke, K.R. 2008. \emph{PERMANOVA+ for PRIMER}: Guide to Software and Statistical Methods. PRIMER-E: Plymouth, UK.

Fletcher, D.J., Underwood, A.J. 2002. How to cope with negative estimates of components of variance in ecological field studies. Journal of Experimental Marine Biology and Ecology 273, 89-95.
}
\seealso{
\code{\link{gad}}, \code{\link{pooling}}, \code{\link{estimates}}
}
\author{
Eliandro Gilbert (\email{eliandrogilbert@gmail.com})
}
