% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reco_lcc.R
\name{cslcc}
\alias{cslcc}
\title{Level conditional coherent reconciliation for genuine hierarchical/grouped time series}
\usage{
cslcc(base, agg_mat, nodes = "auto", comb = "ols", res = NULL, CCC = TRUE,
      const = "exogenous", bts = NULL, approach = "proj", nn = NULL,
      settings = NULL, ...)
}
\arguments{
\item{base}{A (\eqn{h \times n}) numeric matrix or multivariate time series (\code{mts} class)
containing the base forecasts to be reconciled; \eqn{h} is the forecast horizon, and \eqn{n} is
the total number of time series (\eqn{n = n_a + n_b}).}

\item{agg_mat}{A (\eqn{n_a \times n_b}) numeric matrix representing the cross-sectional
aggregation matrix. It maps the \eqn{n_b} bottom-level (free)
variables into the \eqn{n_a} upper (constrained) variables.}

\item{nodes}{A (\eqn{L \times 1}) numeric vector indicating the number of variables
in each of the upper \eqn{L} levels of the hierarchy. The \emph{default}
value is the string "\code{auto}" which calculates the number of variables in each level.}

\item{comb}{A string specifying the reconciliation method. For a complete list, see \link{cscov}.}

\item{res}{An (\eqn{N \times n}) optional numeric matrix containing the in-sample
residuals. This matrix is used to compute some covariance matrices.}

\item{CCC}{A logical value indicating whether the Combined Conditional Coherent reconciled
forecasts reconciliation should include bottom-up forecasts (\code{TRUE}, \emph{default}), or not.}

\item{const}{A string specifying the reconciliation constraints:
\itemize{
\item "\code{exogenous}" (\emph{default}): Fixes the top level of each sub-hierarchy.
\item "\code{endogenous}": Coherently revises both the top and bottom levels.
}}

\item{bts}{A (\eqn{h \times n_b}) numeric matrix or multivariate time series (\code{mts} class)
containing bottom base forecasts defined by the user (e.g., seasonal averages, as in Hollyman et al., 2021).
This parameter can be omitted if only base forecasts are used
(see Di Fonzo and Girolimetto, 2024).}

\item{approach}{A string specifying the approach used to compute the reconciled
forecasts. Options include:
\itemize{
\item "\code{proj}" (\emph{default}): Projection approach according to Byron (1978, 1979).
\item "\code{strc}": Structural approach as proposed by Hyndman et al. (2011).
\item "\code{proj_osqp}": Numerical solution using \href{https://osqp.org/}{\pkg{osqp}}
for projection approach.
\item "\code{strc_osqp}": Numerical solution using \href{https://osqp.org/}{\pkg{osqp}}
for structural approach.
}}

\item{nn}{A string specifying the algorithm to compute non-negative forecasts:
\itemize{
\item "\code{osqp}": quadratic programming optimization
(\href{https://osqp.org/}{\pkg{osqp}} solver).
\item "\code{bpv}": block principal pivoting algorithm.
\item "\code{sntz}": heuristic "set-negative-to-zero" (Di Fonzo and Girolimetto, 2023).
}}

\item{settings}{A list of control parameters.
\itemize{
\item \code{nn = "osqp"} An object of class \code{osqpSettings} specifying settings
for the \href{https://osqp.org/}{\pkg{osqp}} solver. For details, refer to the
\href{https://osqp.org/}{\pkg{osqp} documentation} (Stellato et al., 2020).
\item \code{nn = "bpv"} It includes: \code{ptype} for permutation method ("\code{random}"
or "\code{fixed}", \emph{default}), \code{par} for the number of full exchange rules that
may be attempted (\code{10}, \emph{default}), \code{tol} for the tolerance in convergence
criteria (\code{sqrt(.Machine$double.eps)}, \emph{default}), \code{gtol} for the gradient
tolerance in convergence criteria (\code{sqrt(.Machine$double.eps)}, \emph{default}),
\code{itmax} for the maximum number of algorithm iterations (\code{100}, \emph{default})
}}

\item{...}{
  Arguments passed on to \code{\link[=cscov]{cscov}}
  \describe{
    \item{\code{mse}}{If \code{TRUE} (\emph{default}) the residuals used to compute the covariance
matrix are not mean-corrected.}
    \item{\code{shrink_fun}}{Shrinkage function of the covariance matrix, \link{shrink_estim} (\emph{default}).}
  }}
}
\value{
A (\eqn{h \times n}) numeric matrix of cross-sectional reconciled forecasts.
}
\description{
This function implements the cross-sectional forecast reconciliation procedure that
extends the original proposal by Hollyman et al. (2021). Level conditional coherent
reconciled forecasts are conditional on (i.e., constrained by) the base forecasts
of a specific upper level in the hierarchy (exogenous constraints). It also allows
handling the linear constraints linking the variables endogenously (Di Fonzo and
Girolimetto, 2022). The function can calculate Combined Conditional Coherent (CCC)
forecasts as simple averages of Level-Conditional Coherent (LCC) and bottom-up
reconciled forecasts, with either endogenous or exogenous constraints.
}
\examples{
set.seed(123)
# Aggregation matrix for Z = X + Y, X = XX + XY and Y = YX + YY
A <- matrix(c(1,1,1,1,1,1,0,0,0,0,1,1), 3, byrow = TRUE)
# (2 x 7) base forecasts matrix (simulated)
base <- matrix(rnorm(7*2, mean = c(40, 20, 20, 10, 10, 10, 10)), 2, byrow = TRUE)
# (10 x 7) in-sample residuals matrix (simulated)
res <- matrix(rnorm(n = 7*10), ncol = 7)
# (2 x 7) Naive bottom base forecasts matrix: all forecasts are set equal to 10
naive <- matrix(10, 2, 4)

## EXOGENOUS CONSTRAINTS (Hollyman et al., 2021)
# Level Conditional Coherent (LCC) reconciled forecasts
exo_LC <- cslcc(base = base, agg_mat = A, comb = "wls", bts = naive,
                res = res, nodes = "auto", CCC = FALSE)

# Combined Conditional Coherent (CCC) reconciled forecasts
exo_CCC <- cslcc(base = base, agg_mat = A, comb = "wls", bts = naive,
                 res = res, nodes = "auto", CCC = TRUE)

# Results detailed by level:
# L-1: Level 1 immutable reconciled forecasts for the whole hierarchy
# L-2: Middle-Out reconciled forecasts
# L-3: Bottom-Up reconciled forecasts
info_exo <- recoinfo(exo_CCC, verbose = FALSE)
info_exo$lcc

## ENDOGENOUS CONSTRAINTS (Di Fonzo and Girolimetto, 2024)
# Level Conditional Coherent (LCC) reconciled forecasts
endo_LC <- cslcc(base = base, agg_mat = A, comb = "wls",
                 res = res, nodes = "auto", CCC = FALSE,
                 const = "endogenous")

# Combined Conditional Coherent (CCC) reconciled forecasts
endo_CCC <- cslcc(base = base, agg_mat = A, comb = "wls",
                  res = res, nodes = "auto", CCC = TRUE,
                  const = "endogenous")

# Results detailed by level:
# L-1: Level 1 reconciled forecasts for L1 + L3 (bottom level)
# L-2: Level 2 reconciled forecasts for L2 + L3 (bottom level)
# L-3: Bottom-Up reconciled forecasts
info_endo <- recoinfo(endo_CCC, verbose = FALSE)
info_endo$lcc

}
\references{
Byron, R.P. (1978), The estimation of large social account matrices,
\emph{Journal of the Royal Statistical Society, Series A}, 141, 3, 359-367.
\doi{10.2307/2344807}

Byron, R.P. (1979), Corrigenda: The estimation of large social account matrices,
\emph{Journal of the Royal Statistical Society, Series A}, 142(3), 405.
\doi{10.2307/2982515}

Di Fonzo, T. and Girolimetto, D. (2024), Forecast combination-based forecast reconciliation:
Insights and extensions, \emph{International Journal of Forecasting}, 40(2), 490–514.
\doi{10.1016/j.ijforecast.2022.07.001}

Di Fonzo, T. and Girolimetto, D. (2023b) Spatio-temporal reconciliation of solar forecasts.
\emph{Solar Energy} 251, 13–29. \doi{10.1016/j.solener.2023.01.003}

Hyndman, R.J., Ahmed, R.A., Athanasopoulos, G. and Shang, H.L. (2011),
Optimal combination forecasts for hierarchical time series,
\emph{Computational Statistics & Data Analysis}, 55, 9, 2579-2589.
\doi{10.1016/j.csda.2011.03.006}

Hollyman, R., Petropoulos, F. and Tipping, M.E. (2021), Understanding forecast reconciliation.
\emph{European Journal of Operational Research}, 294, 149–160. \doi{10.1016/j.ejor.2021.01.017}

Stellato, B., Banjac, G., Goulart, P., Bemporad, A. and Boyd, S. (2020), OSQP:
An Operator Splitting solver for Quadratic Programs,
\emph{Mathematical Programming Computation}, 12, 4, 637-672.
\doi{10.1007/s12532-020-00179-2}
}
\seealso{
Level conditional coherent reconciliation: 
\code{\link{ctlcc}()},
\code{\link{telcc}()}

Cross-sectional framework: 
\code{\link{csboot}()},
\code{\link{csbu}()},
\code{\link{cscov}()},
\code{\link{csmo}()},
\code{\link{csrec}()},
\code{\link{cstd}()},
\code{\link{cstools}()}
}
\concept{Framework: cross-sectional}
\concept{Reco: level conditional coherent}
