
#' Define a new simulation in an FSK2R object
#'
#' Sets a new simulation using the parameters defined in simulation_pars.
#' The method updates all the relevant methods.
#'
#' @param fsk_object Instance of FSK2R
#' @param simulation_id A character with an id for the new simulation.
#' @param parameters A list whose names are the parameters to modify and their values
#' their values for the simulation.
#'
#' @export
#'
#' @importFrom purrr set_names
#' @importFrom purrr map
#' @importFrom rlang .data
#'
#' @return An instance of FSK2R with the additional simulation data.
#'
set_new_simulation <- function(fsk_object, simulation_id, parameters) {

    # Helper function to properly format parameter values
    format_parameter_value <- function(param_value) {
        if (is.null(param_value)) {
            return("NULL")
        } else if (is.character(param_value) && length(param_value) == 1) {
            # String literal - store with quotes but NO XML escaping
            # (let XML serializer handle escaping during export)
            return(paste0('"', param_value, '"'))
        } else if (length(param_value) > 1) {
            # Vector or complex object - convert to R expression
            return(deparse(param_value, width.cutoff = 500L))
        } else if (is.logical(param_value) || is.numeric(param_value)) {
            # Simple scalar values
            return(as.character(param_value))
        } else {
            # Other objects - convert to R expression 
            return(deparse(param_value, width.cutoff = 500L))
        }
    }

    # Create changes with R attributes (to match xml2::as_list format)
    changes <- lapply(names(parameters), function(param_name) {
        change_element <- list()
        # Add R attributes (how xml2::as_list stores XML attributes)
        attr(change_element, "newValue") <- format_parameter_value(parameters[[param_name]])
        attr(change_element, "target") <- param_name
        return(change_element)
    })
    names(changes) <- rep("changeAttribute", length(changes))

    # Create new model with R attributes
    new_model <- list(listOfChanges = changes)
    attr(new_model, "id") <- simulation_id
    attr(new_model, "name") <- ""
    attr(new_model, "language") <- "https://iana.org/assignments/mediatypes/text/x-r"
    attr(new_model, "source") <- "./model.r"

    # Check if simulation_id already exists and replace it, otherwise add new
    existing_sims <- get_simulations(fsk_object)
    existing_index <- which(names(existing_sims) == simulation_id)
    
    if (length(existing_index) > 0) {
        # Replace existing simulation
        fsk_object$simulation$sedML$listOfModels[[existing_index[1]]] <- new_model
    } else {
        # Add new simulation
        # Handle different structures from create_fsk vs import_fsk
        if ("model" %in% names(fsk_object$simulation$sedML$listOfModels)) {
            # Structure from create_fsk - convert to list format
            if (length(fsk_object$simulation$sedML$listOfModels) == 1) {
                fsk_object$simulation$sedML$listOfModels <- list(
                    fsk_object$simulation$sedML$listOfModels$model,
                    new_model
                )
            } else {
                fsk_object$simulation$sedML$listOfModels[[length(fsk_object$simulation$sedML$listOfModels) + 1]] <- new_model
            }
        } else {
            # Structure from import_fsk - add to existing list
            fsk_object$simulation$sedML$listOfModels[[length(fsk_object$simulation$sedML$listOfModels) + 1]] <- new_model
        }
        
        # Ensure all models are named consistently
        names(fsk_object$simulation$sedML$listOfModels) <- rep("model", length(fsk_object$simulation$sedML$listOfModels))
    }

    fsk_object
}

#' Run one simulation in an FSK object
#'
#' Runs the simulation corresponding to index. If defined, it also
#' calls any visualization script. Returns all user-created variables
#' from the simulation environment, supporting various data types including
#' scalars, vectors, data frames, lists, and matrices.
#'
#' @importFrom dplyr rename left_join mutate filter tibble
#' @importFrom purrr map_dfr keep
#' @importFrom rlang .data
#' @importFrom tidyr pivot_wider
#'
#' @param fsk_object Instance of FSK2R
#' @param index Index of the simulation
#' @param run_visualization Whether to call the visualization script. FALSE
#' by default.
#' @param copy_workspace Whether to copy the simulation workspace to the user's
#' working directory. FALSE by default.
#' @param workspace_mode What to copy when copy_workspace=TRUE. Options: "all" 
#' (copy everything), "generated" (copy only files created during simulation),
#' "modified" (copy only files modified during simulation). Default is "all".
#' @param inject_to_global Whether to inject simulation variables into the 
#' user's global environment for seamless model chaining. FALSE by default
#' for backward compatibility.
#'
#' @export
#'
#' @return A named list containing all variables created by the simulation model.
#' Each element preserves the original data type (numbers, strings, data frames,
#' lists, matrices, etc.). Returns an empty list if no variables are created.
#' When inject_to_global=TRUE, variables are also available in the global environment.
#'
run_simulation <- function(fsk_object, index, run_visualization = FALSE, copy_workspace = FALSE, workspace_mode = "all", inject_to_global = FALSE) {

    ## Check for missing packages

    # if (length(fsk_object$packages$PackageList) > 0) {
    #
    #     missing_pkgs <- tibble(aa = unlist(fsk_object$packages$PackageList),
    #            bb = names(unlist(fsk_object$packages$PackageList))) %>%
    #         filter(.data$bb == "Package") %>%
    #         filter(!.data$aa %in% rownames(installed.packages()))
    #     missing_pkgs <- missing_pkgs$aa
    #
    #     if (length(missing_pkgs > 0)) install.packages(missing_pkgs, repos = repos)
    #
    # }

    if (length(fsk_object$packages$PackageList) > 0) {

        # Extract package names directly from the list structure (jsonlite/rjson agnostic)
        package_names <- sapply(fsk_object$packages$PackageList, function(x) {
            if (is.list(x) && "Package" %in% names(x)) {
                return(x$Package)
            } else {
                return(NA)
            }
        })
        
        # Filter out any NA values
        package_names <- package_names[!is.na(package_names)]

        if (length(package_names) > 0) {
            # Check if packages can be loaded
            check_pckgs <- sapply(package_names, function(pkg) {
                tryCatch({
                    result <- require(pkg, character.only = TRUE, quietly = TRUE)
                    return(as.logical(result))
                }, error = function(e) {
                    return(FALSE)
                })
            })
            
            # Ensure it's a logical vector
            check_pckgs <- as.logical(check_pckgs)
            
            failed_pckgs <- package_names[!check_pckgs]

            if (length(failed_pckgs) > 0) {
                stop(paste("Some packages required by the FSK model could not be loaded:", 
                          paste(failed_pckgs, collapse = ", "),
                          ". Install them using install.packages() before running the model."))
            }
        }

    }

    ## Execute simulation in temporary directory where all files are available
    
    # Store current working directory to restore later
    original_wd <- getwd()
    
    # Change to the temporary directory where FSKX files were unpacked
    if (!is.null(fsk_object$temp_dir) && dir.exists(fsk_object$temp_dir)) {
        setwd(fsk_object$temp_dir)
    } else {
        warning("Temporary directory not found or not accessible. Files may not be available to the model.")
    }
    
    # Ensure we restore the working directory even if there's an error
    on.exit({
        setwd(original_wd)
    }, add = TRUE)

    ## Define the environment

    sim_env <- new.env(parent = parent.frame())

    if (index > n_simuls_fsk(fsk_object)) stop(paste("Index higher than the number of models"))

    model_item <- fsk_object$simulation$sedML$listOfModels[[index]]
    
    # The actual model content might be nested
    my_model <- if("model" %in% names(model_item)) model_item$model else model_item

    if (is.null(my_model)) stop(paste("Model", index, "not found."))

    ## Assign the parameters for the simulation

    if (length(my_model$listOfChanges) > 0) {

        # Handle metadata safely (may be NULL for create_fsk objects)
        meta_info <- tibble()
        if (!is.null(fsk_object$metadata) && 
            !is.null(fsk_object$metadata$modelMath) &&
            !is.null(fsk_object$metadata$modelMath$parameter)) {
            
            meta_info <- fsk_object$metadata$modelMath$parameter %>%
                map(unlist) %>%
                map(~ {
                    param_names <- names(.)
                    if (is.null(param_names) || length(param_names) == 0) {
                        # Handle parameters without names
                        return(tibble())
                    } else {
                        return(tibble(par = param_names, value = .))
                    }
                }) %>%
                keep(~ nrow(.) > 0) %>%  # Remove empty tibbles
                map_dfr(~ pivot_wider(., names_from = par, values_from = value))
        }

        for (row_n in (1:length(my_model$listOfChanges))) {

            simul_data <- my_model$listOfChanges[[row_n]]
            
            # Handle different structures for attributes
            target <- attr(simul_data, "target")
            newValue <- attr(simul_data, "newValue")
            if (is.null(target)) {
                target <- simul_data$.attrs["target"]
                newValue <- simul_data$.attrs["newValue"]
            }
            
            if (is.na(target) || is.null(target)) next # Skip if target is not defined
            
            # Try to find parameter metadata if id column exists
            if (nrow(meta_info) > 0 && "id" %in% names(meta_info)) {
                par_metadata <- filter(meta_info, .data$id == target)
            } else {
                # No metadata available, continue without it
                par_metadata <- tibble()
            }

            # Unescape XML entities before parsing as R code
            unescaped_value <- gsub("&quot;", '"', newValue)
            unescaped_value <- gsub("&lt;", "<", unescaped_value)
            unescaped_value <- gsub("&gt;", ">", unescaped_value)
            unescaped_value <- gsub("&amp;", "&", unescaped_value)
            
            var_value <- eval(parse(text = unescaped_value), envir = sim_env)
            assign(target, var_value, pos = sim_env)
        }
    }

    ## Capture file state before simulation (for workspace copying)
    files_before <- NULL
    if (copy_workspace && workspace_mode %in% c("generated", "modified")) {
        files_before <- list.files(fsk_object$temp_dir, recursive = TRUE, all.files = TRUE)
        files_before_info <- file.info(file.path(fsk_object$temp_dir, files_before))
    }

    ## Run the simulation

    eval(parse(text = fsk_object$R_model), envir = sim_env)

    if (run_visualization) {
        # Store variables before visualization to detect new ggplot objects
        vars_before_viz <- ls(sim_env)
        
        # Run visualization script
        eval(parse(text = fsk_object$visualization), envir = sim_env)
        
        # Check for new ggplot objects and automatically print them
        vars_after_viz <- ls(sim_env)
        new_vars <- setdiff(vars_after_viz, vars_before_viz)
        
        # Print any new ggplot objects to ensure they display in RStudio
        for(var_name in new_vars) {
            obj <- get(var_name, envir = sim_env)
            if(inherits(obj, "gg")) {  # Check if it's a ggplot object
                print(obj)
            }
        }
        
        # Also try to capture and print any ggplot expressions by re-evaluating the entire visualization
        # This handles cases where ggplot is created but not assigned to a variable (common case)
        tryCatch({
            # Parse the entire visualization script and look for the last expression that returns a ggplot
            parsed_exprs <- parse(text = fsk_object$visualization)
            
            # Evaluate each expression and check if it returns a ggplot object
            for(i in length(parsed_exprs):1) {  # Start from the end
                expr_result <- eval(parsed_exprs[[i]], envir = sim_env)
                if(inherits(expr_result, "gg")) {
                    print(expr_result)
                    break  # Only print the last ggplot found
                }
            }
        }, error = function(e) {
            # Silently continue if there's any issue with expression evaluation
        })
    }

    ## Copy workspace if requested
    if (copy_workspace) {
        # Get model name and simulation name for directory naming
        model_name <- fsk_object$metadata$generalInformation$name
        if (is.null(model_name) || model_name == "") {
            model_name <- "fsk_model"
        }
        
        # Get simulation name
        simulations <- get_simulations(fsk_object)
        simulation_name <- names(simulations)[index]
        if (is.null(simulation_name) || simulation_name == "") {
            simulation_name <- paste0("simulation_", index)
        }
        
        # Sanitize names for directory creation
        model_name <- gsub("[^A-Za-z0-9_-]", "_", model_name)
        simulation_name <- gsub("[^A-Za-z0-9_-]", "_", simulation_name)
        
        # Create workspace directory name
        workspace_dir <- file.path(original_wd, paste0("fsk_workspace_", model_name, "_", simulation_name))
        
        tryCatch({
            # Create workspace directory
            if (!dir.exists(workspace_dir)) {
                dir.create(workspace_dir, recursive = TRUE)
            }
            
            # Always export simulation environment as workspace.r and workspace.RData
            export_workspace(fsk_object, workspace_dir, simulation_env = sim_env)
            
            # Determine which files to copy based on workspace_mode
            files_to_copy <- character(0)
            
            if (workspace_mode == "all") {
                # Copy everything from temp directory
                files_to_copy <- list.files(fsk_object$temp_dir, recursive = TRUE, all.files = TRUE)
                files_to_copy <- files_to_copy[!files_to_copy %in% c(".", "..")]
            } else if (workspace_mode == "generated") {
                # For generated mode, include simulation-generated files plus workspace files
                files_after <- list.files(fsk_object$temp_dir, recursive = TRUE, all.files = TRUE)
                generated_files <- setdiff(files_after, files_before)
                
                # Always include workspace files for "generated" mode
                workspace_files <- c("workspace.r", "workspace.RData")
                files_to_copy <- unique(c(generated_files, workspace_files))
                
                # Also include any plot files that were created
                plot_files <- list.files(fsk_object$temp_dir, pattern = "\\.(pdf|png|jpg|jpeg|svg)$", 
                                       recursive = TRUE, all.files = TRUE)
                files_to_copy <- unique(c(files_to_copy, plot_files))
                
            } else if (workspace_mode == "modified") {
                # Copy files that were created or modified during simulation
                files_after <- list.files(fsk_object$temp_dir, recursive = TRUE, all.files = TRUE)
                files_after_info <- file.info(file.path(fsk_object$temp_dir, files_after))
                
                # Find new files
                new_files <- setdiff(files_after, files_before)
                
                # Find modified files (comparing modification times)
                modified_files <- character(0)
                if (!is.null(files_before)) {
                    common_files <- intersect(files_after, files_before)
                    for (f in common_files) {
                        # Safe comparison that handles NA values in modification times
                        mtime_after <- files_after_info[f, "mtime"]
                        mtime_before <- files_before_info[f, "mtime"]
                        
                        if (!is.na(mtime_after) && !is.na(mtime_before) && 
                            mtime_after > mtime_before) {
                            modified_files <- c(modified_files, f)
                        }
                    }
                }
                
                # Always include workspace files for "modified" mode
                workspace_files <- c("workspace.r", "workspace.RData")
                files_to_copy <- unique(c(new_files, modified_files, workspace_files))
            }
            
            # Copy the selected files
            if (length(files_to_copy) > 0) {
                for (file_path in files_to_copy) {
                    src_file <- file.path(fsk_object$temp_dir, file_path)
                    dest_file <- file.path(workspace_dir, file_path)
                    
                    # Skip if file is workspace.r or workspace.RData (already created by export_workspace)
                    if (basename(file_path) %in% c("workspace.r", "workspace.RData")) {
                        next
                    }
                    
                    # Create destination directory if needed
                    dest_dir <- dirname(dest_file)
                    if (!dir.exists(dest_dir)) {
                        dir.create(dest_dir, recursive = TRUE)
                    }
                    
                    # Copy the file
                    if (file.exists(src_file) && !file.info(src_file)$isdir) {
                        file.copy(src_file, dest_file, overwrite = TRUE)
                    }
                }
                
                message(paste("Workspace copied to:", workspace_dir))
                message(paste("Files copied:", length(files_to_copy)))
            } else {
                message("Workspace exported (no additional files to copy based on workspace_mode)")
            }
            
        }, error = function(e) {
            warning(paste("Failed to copy workspace:", e$message))
        })
    }

    # Return the results - generic approach for any variables
    # Get all variable names from the simulation environment
    all_vars <- ls(sim_env)
    
    # Filter out system objects and functions to keep only user-created variables
    # Exclude: functions, environments, and common R system variables
    user_vars <- all_vars[sapply(all_vars, function(var_name) {
        obj <- get(var_name, envir = sim_env)
        !is.function(obj) && !is.environment(obj) && 
        !var_name %in% c("last.warning", ".Last.value", ".Random.seed")
    })]
    
    if (length(user_vars) > 0) {
        # Collect all user-created variables preserving their types
        results <- mget(user_vars, envir = sim_env)
        
        # Inject variables to global environment if requested
        if (inject_to_global) {
            # Get the global environment (user workspace)
            global_env <- .GlobalEnv
            
            # Inject each variable into the global environment
            for (var_name in user_vars) {
                assign(var_name, get(var_name, envir = sim_env), envir = global_env)
            }
            
            # Inform user about injected variables
            message(paste("Injected", length(user_vars), "simulation variables to global environment:"))
            message(paste("Variables:", paste(user_vars, collapse = ", ")))
        }
        
        return(results)
    } else {
        # Return empty list if no variables found
        return(list())
    }
}

#' Run every simulation in an FSK object
#'
#' Runs every simulation defined in the FSK object. This includes
#' the ones originally included in the FSK container, as well as
#' the ones added using set_new_simulation().
#'
#' @inheritParams run_simulation
#'
#' @export
#'
#' @return A named list with the results of all simulations
#'
run_all_simulations <- function(fsk_object, run_visualization = FALSE, copy_workspace = FALSE, workspace_mode = "all", inject_to_global = FALSE) {

    sim_names <- names(get_simulations(fsk_object))

    results <- lapply(1:n_simuls_fsk(fsk_object), function(i) {
        run_simulation(fsk_object, i, run_visualization, copy_workspace, workspace_mode, inject_to_global)
    })

    set_names(results, sim_names)
}
