
#' @title Control `FDOTT` linear programming solver
#' @description Create an object that controls the linear programming (LP) solver to use.
#' @param name name of the LP solver.
#' @param ... optional control arguments passed to the corresponding LP solver.
#' @details `name` can be any LP solver that is compatible with the `ROI` package infrastructure.
#' In particular, the corresponding plugin package `ROI.plugin.name` must be installed.
#' The default value corresponding to `name = NULL` can be set via `options(FDOTT.lp_solver = name)`
#' (the default is `"glpk"`).
#' @returns A `ot_test_lp_solver_control` object containing:
#' \tabular{ll}{
#'  `name`     \tab the name of the LP solver \cr
#'  `control`   \tab list of control arguments passed to the LP solver \cr
#' }
#' @examples
#' \dontrun{
#' # glpk is already the default
#' options(FDOTT.lp_solver = "glpk")}
#' # plugin needs to be installed, else we get error
#' if (requireNamespace("ROI.plugin.glpk")) {
#'     # add control parameter (specific to glpk)
#'     sol <- ot_test_lp_solver("glpk", verbose = TRUE)
#'     print(sol)
#' } else {
#'     cat("'ROI.plugin.glpk' needs to be installed!\n")
#' }
#' @seealso [`ROI::ROI_available_solvers`]
#' @export
ot_test_lp_solver <- \(name = NULL, ...) {

    stopifnot(is.null(name) || (is_scalar(name) && is.character(name)))

    ls <- list(
        name    = if (is.null(name)) getOption("FDOTT.lp_solver", default = "glpk") else name,
        control = list(...)
    )

    ROI_load(ls$name)

    structure(ls, class = "ot_test_lp_solver_control")
}

is_solver_control <- \(x) methods::is(x, "ot_test_lp_solver_control")

#' @export
print.ot_test_lp_solver_control <- \(x, ...) {
    cat("LP solver: ", x$name, "\n", sep = "")
    if (length(x$control) == 0) {
        cat("With no control parameters.\n")
    } else {
        cat("With control parameters:\n")
        print(x$control)
    }
    invisible(x)
}

##

ROI_load <- \(solver) {
    pkg <- paste0("ROI.plugin.", solver)
    if (!requireNamespace(pkg, quietly = TRUE)) {
        stop(sprintf("LP error: The package '%s' is not available.", pkg))
    }
}

# if using future, then solver = NULL means that glpk will be attached
# (options are not passed by future), so pass it manually
ROI_attach <- \(solver = NULL) {

    if (is.null(solver)) {
        solver <- ot_test_lp_solver()
    }

    ROI_load(solver$name)

    invisible(solver)
}

ROI_check_error <- \(res, add.info = NULL) {
    if (ROI::solution(res, "status_code") == 1) {
        stop("LP error: ", ROI::solution(res, "status")$msg$message, paste0(" [", add.info, "]", recycle0 = TRUE))
    }
}

lp_solve <- \(objective, constraints, types, bounds, maximum, solver = NULL, add.info = NULL) {

    solver <- ROI_attach(solver)

    lp <- ROI::OP(
        objective   = objective,
        constraints = constraints,
        types       = types,
        bounds      = bounds,
        maximum     = maximum
    )

    res <- ROI::ROI_solve(lp, solver$name, control = solver$control)

    ROI_check_error(res, add.info)

    res
}

lp_objval <- \(objective, constraints, types, bounds, maximum, solver = NULL, add.info = NULL) {
    solver <- ROI_attach(solver)

    lp <- ROI::OP(
        objective   = objective,
        constraints = constraints,
        types       = types,
        bounds      = bounds,
        maximum     = maximum
    )

    res <- ROI::ROI_solve(lp, solver$name, control = solver$control)

    if (ROI::solution(res, "status_code") == 1) {
        if (maximum) Inf else -Inf
    } else {
        ROI::solution(res, "objval")
    }
}
