% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dynamic.network.compare.R
\name{dynamic.network.compare}
\alias{dynamic.network.compare}
\title{Compares Dynamic Network Structures Using Permutation}
\usage{
dynamic.network.compare(
  data,
  paired = FALSE,
  corr = c("auto", "cor_auto", "pearson", "spearman"),
  na.data = c("pairwise", "listwise"),
  model = c("BGGM", "glasso", "TMFG"),
  id = NULL,
  group = NULL,
  n.embed = 5,
  n.embed.optimize = FALSE,
  tau = 1,
  delta = 1,
  use.derivatives = 1,
  na.derivative = c("none", "kalman", "rowwise", "skipover"),
  zero.jitter = 0.001,
  iter = 1000,
  ncores,
  seed = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis as well as
an ID column}

\item{paired}{Boolean (length = 1).
Whether groups are repeated measures representing
paired samples.
Defaults to \code{FALSE}}

\item{corr}{Character (length = 1).
Method to compute correlations.
Defaults to \code{"auto"}.
Available options:

\itemize{

\item \code{"auto"} --- Automatically computes appropriate correlations for
the data using Pearson's for continuous, polychoric for ordinal,
tetrachoric for binary, and polyserial/biserial for ordinal/binary with
continuous. To change the number of categories that are considered
ordinal, use \code{ordinal.categories}
(see \code{\link[EGAnet]{polychoric.matrix}} for more details)

\item \code{"cor_auto"} --- Uses \code{\link[qgraph]{cor_auto}} to compute correlations.
Arguments can be passed along to the function

\item \code{"cosine"} --- Uses \code{\link[EGAnet]{cosine}} to compute cosine similarity

\item \code{"pearson"} --- Pearson's correlation is computed for all
variables regardless of categories

\item \code{"spearman"} --- Spearman's rank-order correlation is computed
for all variables regardless of categories

}

For other similarity measures, compute them first and input them
into \code{data} with the sample size (\code{n})}

\item{na.data}{Character (length = 1).
How should missing data be handled?
Defaults to \code{"pairwise"}.
Available options:

\itemize{

\item \code{"pairwise"} --- Computes correlation for all available cases between
two variables

\item \code{"listwise"} --- Computes correlation for all complete cases in the dataset

}}

\item{model}{Character (length = 1).
Defaults to \code{"glasso"}.
Available options:

\itemize{

\item \code{"BGGM"} --- Computes the Bayesian Gaussian Graphical Model.
Set argument \code{ordinal.categories} to determine
levels allowed for a variable to be considered ordinal.
See \code{?BGGM::estimate} for more details

\item \code{"glasso"} --- Computes the GLASSO with EBIC model selection.
See \code{\link[EGAnet]{EBICglasso.qgraph}} for more details

\item \code{"TMFG"} --- Computes the TMFG method.
See \code{\link[EGAnet]{TMFG}} for more details

}}

\item{id}{Numeric or character (length = 1).
Number or name of the column identifying each individual.
Defaults to \code{NULL}}

\item{group}{Numeric or character (length = 1).
Number of the column identifying group membership.
Defaults to \code{NULL}}

\item{n.embed}{Numeric (length = 1).
Defaults to \code{5}.
Number of embedded dimensions (the number of observations to
be used in the \code{\link[EGAnet]{Embed}} function). For example,
an \code{"n.embed = 5"} will use five consecutive observations
to estimate a single derivative}

\item{n.embed.optimize}{Boolean (length = 1).
If \code{TRUE}, performs optimization of \code{n.embed} for each individual,
then constructs the population based on optimized derivatives. When \code{TRUE},
individual networks are considered of interest and will always be output.
Defaults to \code{FALSE}}

\item{tau}{Numeric (length = 1).
Defaults to \code{1}.
Number of observations to offset successive embeddings in
the \code{\link[EGAnet]{Embed}} function.
Generally recommended to leave "as is"}

\item{delta}{Numeric (length = 1).
Defaults to \code{1}.
The time between successive observations in the time series (i.e, lag).
Generally recommended to leave "as is"}

\item{use.derivatives}{Numeric (length = 1).
Defaults to \code{1}.
The order of the derivative to be used in the analysis.
Available options:

\itemize{

\item \code{0} --- No derivatives; consistent with moving average

\item \code{1} --- First-order derivatives; interpreted as "velocity" or
rate of change over time

\item \code{2} --- Second-order derivatives; interpreted as "acceleration" or
rate of the rate of change over time

}

Generally recommended to leave "as is"}

\item{na.derivative}{Character (length = 1).
How should missing data in the embeddings be handled?
Available options (see Boker et al. (2018) in \code{\link[EGAnet]{glla}} references for more details):

\itemize{

\item \code{"none"} (default) --- does nothing and leaves \code{NA}s in data

\item \code{"kalman"} --- uses Kalman smoothing (\code{\link[stats]{KalmanSmooth}}) with
structural time series models (\code{\link[stats]{StructTS}}) to impute missing values.
This approach models the underlying temporal dependencies (trend, seasonality, autocorrelation)
to generate estimates for missing observations while preserving the original time scale.
More computationally intensive than the other methods but typically provides the
most accurate imputation by respecting the stochastic properties of the time series

\item \code{"rowwise"} --- adjusts time interval with respect to each embedding ensuring
time intervals are adaptive to the missing data (tends to be more accurate than \code{"none"})

\item \code{"skipover"} --- "skips over" missing data and treats the non-missing points
as continuous points in time (note that the time scale shifts to the "per mean time interval,"
which is different and \emph{larger} than the original scale)

}}

\item{zero.jitter}{Numeric (length = 1).
Small amount of Gaussian noise added to zero variance derivatives to prevent
estimation failures. For more than one variable, noise is generated
multivariate normal distribution to ensure orthogonal noise is added.
The jitter preserves the overall structure but avoids singular
covariance matrices during network estimation.
Defaults to \code{0.001}}

\item{iter}{Numeric (length = 1).
Number of permutations to perform.
Defaults to \code{1000} (recommended)}

\item{ncores}{Numeric (length = 1).
Number of cores to use in computing results.
Defaults to \code{ceiling(parallel::detectCores() / 2)} or half of your
computer's processing power.
Set to \code{1} to not use parallel computing}

\item{seed}{Numeric (length = 1).
Defaults to \code{NULL} or random results.
Set for reproducible results.
See \href{https://r-ega.net/articles/reproducibility-prng.html}{Reproducibility and PRNG}
for more details on random number generation in \code{\link{EGAnet}}}

\item{verbose}{Boolean (length = 1).
Should progress be displayed?
Defaults to \code{TRUE}.
Set to \code{FALSE} to not display progress}

\item{...}{Additional arguments that can be passed on to
\code{\link[EGAnet]{auto.correlate}},
\code{\link[EGAnet]{network.estimation}},
\code{\link[EGAnet]{EGA}}, and
\code{\link[EGAnet]{jsd}}}
}
\value{
Returns a list:

\item{network}{Data frame with row names of each measure, empirical value (\code{statistic}), and
\emph{p}-value based on the permutation test (\code{p.value})}

\item{edges}{List containing matrices of values for empirical values (\code{statistic}),
\emph{p}-values (\code{p.value}), and Benjamini-Hochberg corrected \emph{p}-values (\code{p.adjusted})}
}
\description{
A permutation implementation to determine statistical
significance of whether the dynamic network structures are different from one another
}
\examples{
# Three similar groups

# Set seed
set.seed(42)

# Simulate dynamic data
participants <- lapply(
  seq_len(50), function(i){

    # Get output
    output <- simDFM(
      variab = 6, timep = 15,
      nfact = 2, error = 0.100,
      dfm = "DAFS", loadings = 0.60,
      autoreg = 0.80, crossreg = 0.10,
      var.shock = 0.36, cov.shock = 0.18,
      burnin = 2000
    )

    #  Add ID
    df <- data.frame(
      ID = i,
      Group = rep(1:3, each = 5),
      output$data
    )

    # Return data
    return(df)

  }
)

# Put participants into a data frame
df <- do.call(rbind.data.frame, participants)

\dontrun{
# Perform comparison
dynamic.network.compare(
  data = df, paired = TRUE,
  # EGA arguments
  corr = "auto", na.data = "pairwise", model = "glasso",
  # dynEGA arguments
  id = "ID", group = "Group", n.embed = 3,
  tau = 1, delta = 1, use.derivatives = 1,
  # Permutation arguments
  iter = 1000, ncores = 2, verbose = TRUE, seed = 42
)}

# Two similar groups and one different

# Simulate dynamic data
participants <- lapply(
  seq_len(50), function(i){

    # Get output
    output <- simDFM(
      variab = 4, timep = 5,
      nfact = 3, error = 0.100,
      dfm = "DAFS", loadings = 0.60,
      autoreg = 0.80, crossreg = 0.10,
      var.shock = 0.36, cov.shock = 0.18,
      burnin = 2000
    )

    #  Add ID
    df <- data.frame(
      ID = i,
      Group = rep(3, each = 5),
      output$data
    )

    # Return data
    return(df)

  }
)

# Replace group 3
new_group <- do.call(rbind.data.frame, participants)
df[df$Group == 3,] <- new_group

\dontrun{
# Perform comparison
dynamic.network.compare(
  data = df, paired = TRUE,
  # EGA arguments
  corr = "auto", na.data = "pairwise", model = "glasso",
  # dynEGA arguments
  id = "ID", group = "Group", n.embed = 3,
  tau = 1, delta = 1, use.derivatives = 1,
  # Permutation arguments
  iter = 1000, ncores = 2, verbose = TRUE, seed = 42
)}

}
\references{
\strong{Frobenius Norm} \cr
Ulitzsch, E., Khanna, S., Rhemtulla, M., & Domingue, B. W. (2023).
A graph theory based similarity metric enables comparison of subpopulation psychometric networks.
\emph{Psychological Methods}.

\strong{Jensen-Shannon Similarity (1 - Distance)} \cr
De Domenico, M., Nicosia, V., Arenas, A., & Latora, V. (2015).
Structural reducibility of multilayer networks.
\emph{Nature Communications}, \emph{6}(1), 1–9.

\strong{Total Network Strength} \cr
van Borkulo, C. D., van Bork, R., Boschloo, L., Kossakowski, J. J., Tio, P., Schoevers, R. A., Borsboom, D., & Waldorp, L. J. (2023).
Comparing network structures on three aspects: A permutation test.
\emph{Psychological Methods}, \emph{28}(6), 1273–1285.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
