#' Two-Sample Covariance Test for High-Dimensional Data
#'
#' Performs a test to compare the covariance matrices of two high-dimensional samples.
#' This test is designed for situations where the number of variables \( p \) is large relative to the  sample sizes \( n_1 \) and \( n_2 \).
#'
#' The test statistic is based on the maximum normalized squared difference between the two 
#' sample covariance matrices.
#' The p-value is computed using an extreme value distribution.
#'
#' @param X A numeric matrix representing the first sample, where rows are observations and 
#' columns are variables.
#' @param Y A numeric matrix representing the second sample, where rows are observations 
#' and columns are variables.
#'
#' @return A list containing:
#' \item{stat}{The test statistic.}
#' \item{pval}{The p-value of the test.}
#'
#' @examples
#' # Example usage:
#' set.seed(123)
#' n1 <- 20
#' n2 <- 20
#' p <- 50
#' X <- matrix(rnorm(n1 * p), nrow = n1, ncol = p)
#' Y <- matrix(rnorm(n2 * p), nrow = n2, ncol = p)
#' result <- covclx(X, Y)
#' print(result)
#'
#' @export
covclx <- function(X,Y)
{
  n1=nrow(X);n2=nrow(Y);p=ncol(X); p2=ncol(Y)
  Sigma.hat.1=((n1-1)/n1)*cov(X)
  Sigma.hat.2=((n2-1)/n2)*cov(Y)
  X.c=scale(X);Y.c=scale(Y);
  Theta.hat.1=matrix(NA,p,p)
  Theta.hat.1=(1/n1)*t(X.c^2)%*%(X.c^2)-(2/n1)*(Sigma.hat.1)*(t(X.c)%*%(X.c))+Sigma.hat.1^2
  Theta.hat.2=matrix(NA,p,p)
  Theta.hat.2=(1/n2)*t(Y.c^2)%*%(Y.c^2)-(2/n2)*(Sigma.hat.2)*(t(Y.c)%*%(Y.c))+Sigma.hat.2^2
  M.value=max(((Sigma.hat.1- Sigma.hat.2)^2)/(Theta.hat.1/n1+Theta.hat.2/n2))
# 
pgumbel <- function(q, loc = 0, scale = 1) {
  return(exp(-exp(-(q - loc) / scale)))
}

# 
F.extreme.cov <- function(stat, loc = 0, scale = 1) {
  return(pgumbel(stat, loc = loc, scale = scale))
}
  stat_cov_CLX = M.value-4*log(p)+log(log(p))
  pval_cov_CLX = 1-F.extreme.cov(stat_cov_CLX)
  return(list(stat = stat_cov_CLX,pval = pval_cov_CLX))
}
