% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/max_qEI.R
\name{max_qEI}
\alias{max_qEI}
\title{Maximization of multipoint expected improvement criterion (qEI)}
\usage{
max_qEI(
  model,
  npoints,
  lower,
  upper,
  crit = "exact",
  minimization = TRUE,
  optimcontrol = NULL
)
}
\arguments{
\item{model}{an object of class \code{\link[DiceKriging]{km}} ,}

\item{npoints}{an integer representing the desired number of iterations,}

\item{lower}{vector of lower bounds,}

\item{upper}{vector of upper bounds,}

\item{crit}{"exact", "CL" : a string specifying the criterion used. "exact"
triggers the maximization of the multipoint expected improvement at each
iteration (see \code{\link{max_qEI}}), "CL" applies the Constant Liar
heuristic,}

\item{minimization}{logical specifying if the qEI to be maximized is used
in minimiziation or in maximization,}

\item{optimcontrol}{an optional list of control parameters for
optimization. See details.}
}
\value{
A list with components:

\item{par}{A matrix containing the \code{npoints} input vectors found.}

\item{value}{A value giving the qEI computed in \code{par}.}
}
\description{
Maximization of the \code{\link{qEI}} criterion. Two options are available
: Constant Liar (CL), and brute force qEI maximization with
Broyden-Fletcher-Goldfarb-Shanno (BFGS) algorithm, or GENetic Optimization
Using Derivative (genoud) algorithm.
}
\details{
- CL is a heuristic method. First, the regular Expected Improvement EI is
maximized (\code{\link{max_EI}}). Then, for the next points, the Expected
Improvement is maximized again, but with an artificially updated Kriging
model. Since the response values corresponding to the last best point
obtained are not available, the idea of CL is to replace them by an
arbitrary constant value L (a "lie") set by the user (default is the
minimum of all currently available observations).

- The BFGS algorithm is implemented in the standard function
\code{\link{optim}}. Analytical formulae of \code{\link{qEI}} and its
gradient \code{\link{qEI.grad}} are used. The \code{nStarts} starting
points are by default sampled with respect to the regular EI
(\code{\link{sampleFromEI}}) criterion.

- The "genoud" method calls the function \code{\link[rgenoud]{genoud}} using
analytical formulae of \code{\link{qEI}} and its gradient
\code{\link{qEI.grad}}.

The parameters of list \code{optimcontrol} are :

- \code{optimcontrol$method} : "BFGS" (default), "genoud" ; a string
specifying the method used to maximize the criterion (irrelevant when
\code{crit} is "CL" because this method always uses genoud),

- when \code{crit="CL"} :

+ \code{optimcontrol$parinit} : optional matrix of initial values (must
have model@d columns, the number of rows is not constrained),

+ \code{optimcontrol$L} : "max", "min", "mean" or a scalar value specifying
the liar ; "min" takes \code{model@min}, "max" takes \code{model@max},
"mean" takes the prediction of the model ; When L is \code{NULL}, "min" is
taken if \code{minimization==TRUE}, else it is "max".

+ The parameters of function \code{\link[rgenoud]{genoud}}. Main parameters are :
\code{"pop.size"} (default : [N=3*2^model@d for dim<6 and N=32*model@d
otherwise]), \code{"max.generations"} (default : 12),
\code{"wait.generations"} (default : 2) and \code{"BFGSburnin"} (default :
2).

- when \code{optimcontrol$method = "BFGS"} :

+ \code{optimcontrol$nStarts} (default : 4),

+ \code{optimcontrol$fastCompute} : if TRUE (default), a fast approximation
method based on a semi-analytic formula is used, see [Marmin 2014] for
details,

+ \code{optimcontrol$samplingFun} : a function which sample a batch of
starting point (default : \code{\link{sampleFromEI}}),

+ \code{optimcontrol$parinit} : optional 3d-array of initial (or candidate)
batches (for all \code{k}, parinit[,,k] is a matrix of size
\code{npoints*model@d} representing one batch). The number of initial
batches (length(parinit[1,1,])) is not contrained and does not have to be
equal to \code{nStarts}. If there is too few initial batches for
\code{nStarts}, missing batches are drawn with \code{samplingFun} (default
: \code{NULL}),

- when \code{optimcontrol$method = "genoud"} :

+ \code{optimcontrol$fastCompute} : if TRUE (default), a fast approximation
method based on a semi-analytic formula is used, see [Marmin 2014] for
details,

+ \code{optimcontrol$parinit} : optional matrix of candidate starting
points (one row corresponds to one point),

+ The parameters of the \code{\link[rgenoud]{genoud}} function. Main parameters are
\code{"pop.size"} (default : \code{[50*(model@d)*(npoints)]}),
\code{"max.generations"} (default : 5), \code{"wait.generations"} (default
: 2), \code{"BFGSburnin"} (default : 2).
}
\examples{



set.seed(000)
# 3-points EI maximization.
# 9-points factorial design, and the corresponding response
d <- 2
n <- 9
design.fact <- expand.grid(seq(0,1,length=3), seq(0,1,length=3)) 
names(design.fact)<-c("x1", "x2")
design.fact <- data.frame(design.fact) 
names(design.fact)<-c("x1", "x2")
response.branin <- apply(design.fact, 1, branin)
response.branin <- data.frame(response.branin) 
names(response.branin) <- "y" 
lower <- c(0,0)
upper <- c(1,1)

# number of point in the bacth
batchSize <- 3

# model identification
fitted.model <- km(~1, design=design.fact, response=response.branin, 
                   covtype="gauss", control=list(pop.size=50,trace=FALSE), parinit=c(0.5, 0.5))

# maximization of qEI

# With a multistarted BFGS algorithm
maxBFGS <- max_qEI(model = fitted.model, npoints = batchSize, lower = lower, upper = upper, 
crit = "exact",optimcontrol=list(nStarts=3,method = "BFGS"))

# comparison
print(maxBFGS$value)
\dontrun{
# With a genetic algorithme using derivatives
maxGen  <- max_qEI(model = fitted.model, npoints = batchSize, lower = lower, upper = upper, 
crit = "exact", optimcontrol=list(nStarts=3,method = "genoud",pop.size=100,max.generations = 15))
# With the constant liar heuristic
maxCL   <- max_qEI(model = fitted.model, npoints = batchSize, lower = lower, upper = upper, 
crit = "CL",optimcontrol=list(pop.size=20))
print(maxGen$value)
print(maxCL$value)
}


}
\references{
C. Chevalier and D. Ginsbourger (2014) Learning and Intelligent
Optimization - 7th International Conference, Lion 7, Catania, Italy,
January 7-11, 2013, Revised Selected Papers, chapter Fast computation of
the multipoint Expected Improvement with applications in batch selection,
pages 59-69, Springer.

D. Ginsbourger, R. Le Riche, L. Carraro (2007), A Multipoint Criterion for
Deterministic Parallel Global Optimization based on Kriging. The
International Conference on Non Convex Programming, 2007.

D. Ginsbourger, R. Le Riche, and L. Carraro. Kriging is well-suited to
parallelize optimization (2010), In Lim Meng Hiot, Yew Soon Ong, Yoel
Tenne, and Chi-Keong Goh, editors, \emph{Computational Intelligence in
Expensive Optimization Problems}, Adaptation Learning and Optimization,
pages 131-162. Springer Berlin Heidelberg.

J. Mockus (1988), \emph{Bayesian Approach to Global Optimization}. Kluwer
academic publishers.

M. Schonlau (1997), \emph{Computer experiments and global optimization},
Ph.D. thesis, University of Waterloo.
}
\seealso{
\code{\link{qEI}}, \code{\link{qEI.grad}}
}
\author{
Sebastien Marmin 

Clement Chevalier 

David Ginsbourger
}
\keyword{optimize}
