#' @title Distributed Gul Principal Component Analysis
#' @description Performs distributed Gul-type principal component analysis
#' on a numeric dataset split across multiple nodes.
#'
#' @param data A numeric matrix containing the total dataset.
#' @param m An integer specifying the number of principal components for the first stage.
#' @param n1 An integer specifying the length of each data subset.
#' @param K An integer specifying the number of nodes.
#'
#' @return A list with the following components:
#' \describe{
#'   \item{AU1}{List of estimated first-stage loading matrices for each node.}
#'   \item{AU2}{List of estimated second-stage loading matrices for each node.}
#'   \item{DU3}{List of diagonal residual variance matrices for each node.}
#'   \item{shat}{List of covariance matrices of reconstructed data for each node.}
#' }
#'
#' @examples
#' set.seed(123)
#' data <- matrix(rnorm(500), nrow = 100, ncol = 5)
#' DGulPC(data = data, m = 3, n1 = 20, K = 5)
#'
#' @export
DGulPC <- function(data, m, n1, K) {
  SigmaU1hat <- list()
  SigmaU2hat <- list()
  AU1 <- list()
  AU2 <- list()
  DU1 <- list()
  DU2 <- list()
  DU3 <- list()
  F1hat <- list()
  Fhat <- list()
  shat <- list()
  
  for (i in 1:K) {
    n <- nrow(data)
    p <- ncol(data)
    pc <- 2
    L <- matrix(rep(0, K * n1), ncol = n1)
    R <- matrix(0, n1, n)
    L[i, ] <- sample(1:n, n1, replace = FALSE)
    r <- matrix(c(1:n1, L[i, ]), ncol = n1, byrow = TRUE)
    R[t(r)] <- 1
    X1 <- R %*% as.matrix(data)
    X <- scale(X1)
    
    SigmaU1hat[[i]] <- cor(X)
    eig5 <- eigen(SigmaU1hat[[i]])
    lambda1hat <- eig5$values[1:m]
    ind <- order(lambda1hat, decreasing = TRUE)
    Q1 <- eig5$vectors[, ind]
    AU11 <- matrix(0, nrow = p, ncol = m)
    for (j in 1:m) {
      AU11[, j] <- sqrt(lambda1hat[j]) * Q1[, j]
    }
    AU1[[i]] <- AU11
    hU1 <- diag(AU1[[i]] %*% t(AU1[[i]]))
    DU1[[i]] <- diag(SigmaU1hat[[i]] - hU1)
    
    F1hat[[i]] <- X %*% AU1[[i]]
    F1star <- F1hat[[i]] / sqrt(n)
    SigmaU2hat[[i]] <- cor(F1star)
    eig6 <- eigen(SigmaU2hat[[i]])
    lambda2hat <- eig6$values[1:pc]
    ind <- order(lambda2hat, decreasing = TRUE)
    Q2 <- eig6$vectors[, ind]
    AU22 <- matrix(0, nrow = m, ncol = pc)
    for (j in 1:pc) {
      AU22[, j] <- sqrt(lambda2hat[j]) * Q2[, j]
    }
    AU2[[i]] <- AU22
    hU2 <- diag(AU2[[i]] %*% t(AU2[[i]]))
    DU2[[i]] <- diag(SigmaU2hat[[i]] - hU2)
    
    Fhat[[i]] <- F1star %*% AU2[[i]]
    Xhat <- Fhat[[i]] %*% t(AU2[[i]]) %*% t(AU1[[i]])
    shat[[i]] <- cov(Xhat)
    hU3 <- diag(t(t(AU2[[i]]) %*% t(AU1[[i]])) %*% (t(AU2[[i]]) %*% t(AU1[[i]])))
    DU3[[i]] <- diag(shat[[i]] - hU3)
  }
  
  return(list(AU1 = AU1, AU2 = AU2, DU3 = DU3, shat = shat))
}