# ---------- power-user init evaluation ----------
#' Evaluate initial parameter values for mixture fitting
#'
#' @param par_init numeric vector of initial parameters
#' @param x numeric vector of data
#' @param family "lognormal" or "normal"
#' @param lower numeric vector of lower bounds
#' @param upper numeric vector of upper bounds
#' @param pgtol numeric, gradient tolerance for optim
#' @return list with success flag, optimized parameters, log-likelihood, and convergence
#' @export
evaluate_init <- function(par_init, x, family = c("lognormal", "normal"),
                          lower = NULL, upper = NULL, pgtol = 1e-8) {
  family <- match.arg(family)
  if (family == "lognormal") {
    if (is.null(lower) || is.null(upper)) bb <- default_bounds_lognorm2(x)
  } else {
    if (is.null(lower) || is.null(upper)) bb <- default_bounds_norm2(x)
  }
  if (is.null(lower)) lower <- bb$lower
  if (is.null(upper)) upper <- bb$upper
  loglik_fn <- if (family == "lognormal") loglik_lognorm else loglik_norm
  res_opt <- tryCatch({
    optim(par_init, fn = function(par) -loglik_fn(par, x),
          method = "L-BFGS-B", lower = lower, upper = upper,
          control = list(factr = 1e7, pgtol = pgtol, maxit = 100000))
  }, error = function(e) e)
  if (inherits(res_opt, "error")) list(success = FALSE, message = res_opt$message)
  else list(success = (res_opt$convergence == 0), par = res_opt$par,
            logLik = -res_opt$value, convergence = res_opt$convergence, message = res_opt$message)
}
