% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PanelEstimation.R
\name{PanelEstimation}
\alias{PanelEstimation}
\title{Variance estimation for panel data}
\usage{
PanelEstimation(
  x,
  numerator,
  denominator = NULL,
  linComb = matrix(0, 0, n),
  linComb0 = NULL,
  estType = "robustModel",
  leveragePower = 1/2,
  group = NULL,
  returnCov = FALSE,
  usewGross = TRUE
)
}
\arguments{
\item{x}{Output from wideFromCalibrate.}

\item{numerator}{y variable name or number.}

\item{denominator}{y variable name or number.}

\item{linComb}{Matrix defining linear combinations of waves.}

\item{linComb0}{Linear combination matrix to be used prior to ratio
calculations.}

\item{estType}{Estimation type: "robustModel" (default), "ssbAKU",
"robustModelww", "robustModelGroup" or "robustModelGroupww" (see below)}

\item{leveragePower}{Power used when adjusting residuals using leverages.}

\item{group}{Extra variable name or number for cluster robust estimation.}

\item{returnCov}{Return covariance matrices instead of variance vectors.}

\item{usewGross}{Use wGross (if avaliable) instead of design weights to
adjust covariance matrix in the case of NA popTotals}
}
\value{
\item{wTot}{Sum of weights} \item{estimates}{Ordinary estimates}
\item{linCombs}{Estimates of linear combinations}
\item{varEstimates}{Variance of estimates} \item{varLinCombs}{Variance of
estimates of linear combinations} When denominator is specified the above
output refer to ratios. Then, similar output for numerator and denominator
are also included.
}
\description{
Variance estimation of linear combinations of totals and ratios based on
output from wideFromCalibrate
}
\details{
When denominator=NULL, only estimates for a single y-variable (numerator)
are calculated. When denominator is specified, estimates for numerator,
denominator and ratio are calculated. The default estimation type parameter,
"robustModel", is equation (12) in paper. "ssbAKU" is (16), "robustModelww"
is (9) and "robustModelGroup" and "robustModelGroupww" are cluster robust
variants based on \eqn{(w-1)^2} and \eqn{w^2} .
}
\examples{

# Generates data  - two years
z    = AkuData(3000)  # 3000 in each quarter
zPop = AkuData(10000)[,1:7]

# Calibration and "WideFromCalibrate"
b = CalibrateSSB(z,calmodel="~ sex*age", partition=c("year","q"),
        popData=zPop, y=c("unemployed","workforce"))
bWide = WideFromCalibrate(b,CrossStrata(z[,c("year","q")]),z$id)

# Define linear combination matrix
lc = rbind(LagDiff(8,4),PeriodDiff(8,4))
rownames(lc) = c("diffQ1","diffQ2","diffQ3","diffQ4","diffYearMean")
colnames(lc) = colnames(head(bWide$y[[1]]))
lc

# Unemployed: Totals and linear combinations
d1=PanelEstimation(bWide,"unemployed",linComb=lc)  #

# Table of output
cbind(tot=d1$estimates,se=sqrt(d1$varEstimates))
cbind(tot=d1$linCombs,se=sqrt(d1$varLinCombs))

# Ratio: Totals and linear combinations
d=PanelEstimation(bWide,numerator="unemployed",denominator="workforce",linComb=lc)
cbind(tot=d$estimates,se=sqrt(d$varEstimates))
cbind(tot=d$linCombs,se=sqrt(d$varLinCombs))

\dontrun{
# Calibration when som population totals unknown (edu)
# Leverages in output (will be used to adjust residuals)
# Cluster robust estimation (families/famid)
b2 = CalibrateSSB(z,popData=zPop,calmodel="~ edu*sex + sex*age",
           partition=c("year","q"), y=c("unemployed","workforce"),
           leverageOutput=TRUE)
b2Wide = WideFromCalibrate(b2,CrossStrata(z[,c("year","q")]),z$id,extra=z$famid)
d2 = PanelEstimation(b2Wide,"unemployed",linComb=lc,group=1,estType = "robustModelGroup")
cbind(tot=d2$linCombs,se=sqrt(d2$varLinCombs))
}


# Yearly mean before ratio calculation (linComb0)
# and difference between years (linComb)
g=PanelEstimation(bWide,numerator="unemployed",denominator="workforce",
    linComb= LagDiff(2),linComb0=Period(8,4))
cbind(tot=g$linCombs,se=sqrt(g$varLinCombs))

}
\seealso{
\code{\link{CalibrateSSB}}, \code{\link{CalSSBobj}}, \code{\link{WideFromCalibrate}}, \code{\link{CalibrateSSBpanel}}.
}
