#' validateSegData
#'
#' This function is used to validate and clean up all of the input data. It 
#' converts all the columns to numeric that need to be, and filters out
#' any invalid segments, like ones with 0 or negative length. It also converts
#' the chromosome tags to the proper format for feature extraction.
#'
#' @param segData The copy number segment data
#' @param cores The number of cores to use for parallel processing. Default 1.
#' @return list of dataframes containing converted seg data
#' @keywords validate segData
#' @import foreach
#' @export
#' @examples
#' validateSegData(segDataExp)
validateSegData = function(segData, cores = 1){
  
  workers = parallel::makeCluster(cores, type = "SOCK")
  doParallel::registerDoParallel(workers)
  
  numericColumns = c("start","end","segVal")
  
  i=1
  #Assigns each feature to a different worker
  temp_list = foreach(i = 1:length(segData)) %dopar% {
    segSub = segData[[i]]
    
    # Makes sure the start end and segVal are numeric
    segSub[,numericColumns] = apply(segSub[,numericColumns],2,as.numeric)
    
    if (!is.integer(segSub$segVal)){
      segSub$segVal = as.integer(round(segSub$segVal))
    }
    
    # Removes spurious negative segVals
    segSub$segVal[segData$segVal<0] = 0
    
    # Removes segments with negative or zero lengths
    segSub = segSub[segSub$end > segSub$start,]
    
    segSub$chromosome[segSub$chromosome == "23"] = "X"
    segSub$chromosome = paste0("chr",segSub$chromosome)
    segSub
    
  }
  
  parallel::stopCluster(workers)
  names(temp_list) = names(segData)
  return(temp_list)
}
