#' sigSim
#'
#' This function is used to compare two sets of signatures by finding the 
#' similarity matrix across both signature sets. If the signatures have the same
#' underlying components similarity is calculated using the cosine similarity.
#' If the signatures have different underlying components the similarity is
#' estimated using a ks-statistic based measure. See package vignette for more 
#' information.
#'
#' @param reference Results from your reference analysis
#' @param toCompare Results from run that you want to compare to reference
#' @param plot If T, displays the heatmap plot
#' @param text If T, displays the similarity value on the plot
#' @return Plots signature similarity and returns the avg similarity
#' @export
#' @examples
#' sigSim(referenceExp, referenceExp)
sigSim = function(reference, toCompare, plot = TRUE, text = TRUE){
   
   sigMatRef = reference[["sigs"]]
   sigMatComp = toCompare[["sigs"]]
   
   refComps = reference[["CN_components"]]
   compComps = toCompare[["CN_components"]]
   
   if (!identical(names(refComps),names(compComps))){
      warning("Results have different components used. Only using overlapping components for comparison.")
      overlappingCompNames = intersect(names(refComps),names(compComps))
      
      refComps = refComps[overlappingCompNames]
      compComps = compComps[overlappingCompNames]
      
      sigMatRef = sigMatRef[grepl(paste(overlappingCompNames,collapse="|"),rownames(sigMatRef)),]
      sigMatComp = sigMatComp[grepl(paste(overlappingCompNames,collapse="|"),rownames(sigMatComp)),]
      
   }
   

   # Checks whether or not the components are the same
   sameComps = identical(refComps,compComps)
   
   nSigRef = ncol(sigMatRef)
   nSigComp = ncol(sigMatComp)

   cors = matrix(0,nSigRef,nSigComp)
   cosSim = function(x,y){return(sum(x*y)/(sqrt(sum(x^2))*sqrt(sum(y^2))))}

   #Goes through each of the signatures
   for (i in 1:nSigRef){
      for (j in 1:nSigComp){
         if (sameComps) {
            cors[i,j] = cosSim(sigMatRef[,i],sigMatComp[,j])
         } else {
            cors[i,j] = diffCompSigSim(refComps,sigMatRef[,i],compComps,sigMatComp[,j])
         }
      }
   }

   cors = round(cors,6)
   rownames(cors) = paste0("RefSig",1:nSigRef)
   colnames(cors) = paste0("CompSig",1:nSigComp)
   
   if (plot){
      myColors = grDevices::rgb((grDevices::colorRamp(c("white", "#FFBDBD", "red")))(0:49/50)/255)
      pheatmap::pheatmap(cors, color = myColors, breaks = seq(0, 1, 0.02), 
                         cluster_cols = FALSE, cluster_rows = FALSE, 
                         main = "Signature Similarity", display_numbers = text,
                         number_format = "%.3f",fontsize_number = 10,
                         border_color = NA,number_color = "black",angle_col = 0)
   }
   return(cors)
}
