#' detSigNumPipeline
#'
#' This function allows you to run the Copy number signature pipeline
#' up until the determineSigNum call. This is useful if you want to repeteadly
#' check the optimal number of signatures for different sample sets.
#' May take a while, especially if not given multiple cores. 
#' @param segData The data to be analyzed. If a path name, readSegs is used to
#' make the list. Otherwise the list must be formatted correctly. Refer to 
#' ?readSegs for format information.
#' @param cores The number of computer cores to be used for parallel processing
#' @param components Can be used when fixing components. Default is NULL.
#' @param saveRes Whether or not to save the resulting tables and plots. 
#' Default is FALSE
#' @param runName Used to title plots and files when saving results
#' @param rmin Minimum number of signatures to look for. Default is 3.
#' @param rmax Maximum number of signatures to look for. Default is 12.
#' @param max_comps vector of length 6 specifying the max number of components 
#' for each feature. Passed to fitModels. Default is 10 for all features
#' @param min_comps vector of length 6 specifying the min number of components
#'  for each feature. Passed to fitModels. Default is 2 for all features
#' @param saveDir Used to specify where to save the results, must be provided 
#' if using saveDir 
#' @param smooth Whether or not to smooth the input data. Default is FALSE.
#' @param colMap Mapping of column names when reading from text file. 
#' Default column names are ID, chromosome, start, end, segVal.
#' @param pR Peak Reduction
#' @param gbuild The reference genome build. Default is hg19. Also supports hg18 and hg38.
#' @param featsToUse Vector of feature names that you wish to use
#' @param ploidyData The ploidy data to use as a feature
#' @keywords detSigNum
#' @return  Returns a list with all of the results from the pipeline
#' @export
#' @examples
#' #Runs the entire pipeline on the example data giving it 6 cores and specifying
#' #5 signatures with a name of "TCGA Test"
#' \donttest{
#' detSigNumPipeline(segDataExp, cores = 6, saveRes = FALSE, runName = "TCGA Test")}
detSigNumPipeline = function(segData, cores = 1,components=NULL,  saveRes = FALSE, 
                        runName = "Run", rmin = 3, rmax = 12, max_comps = NULL, 
                        min_comps = NULL, saveDir = NULL,
                        smooth = FALSE, colMap = NULL,pR=FALSE,gbuild = "hg19",
                        featsToUse = NULL, ploidyData = NULL){
   results = list()
   results[["func"]] = sys.call()
   
   #Enforces default features if none were specified
   if (is.null(featsToUse)){
     featsToUse = CNSigs::defaultFeats;
   }
   
   # Checks the ploidy related parameters to see if it needs to read the data in
   readPloidy = FALSE
   if ("ploidy" %in% featsToUse){
     readPloidy = is.null(ploidyData)
   } 
   else if (!is.null(ploidyData)) {
     featsToUse = c(featsToUse,"ploidy")
   }
   
   # Sets up the segData either using readSegs or making sure it is a list that
   # is formatted correctly
   if (is.character(segData)){
      segData = readSegs(segData, colMap, readPloidy)
   }
   else if (!is.list(segData)){
      stop("Input segData must be in a list format. Refer to ?readSegs for specifics on data format")
   }
   else if (!identical(names(segData[[1]]),c("chromosome","start","end","segVal"))){
      msg = paste0("Input segData column names not correct.\n  ",
                   "Expected: chromosome, start, end, segVal\n  ",
                   "Got: ", paste(names(segData[[1]]),collapse = ", "),"\n  ",
                   "Refer to ?readSegs for specifics on data format")
      stop(msg)
   }
   
   if (readPloidy){
      ploidyData = segData[[2]]
      segData = segData[[1]]
   }
   
   message("Using ",length(segData)," samples for the analysis.\n")
   
   # Smooths the data if specified
   if (smooth){
      results[["Raw Input"]] = segData
      message("Smoothing Segments.\n")
      segData = smoothSegs(segData, cores)
      results[["Input_data"]] = segData
   }
   else {
      results[["Input_data"]] = segData
   }
   
   #When saveRes=TRUE it makes a new folder with runName and the date to save to
   if (saveRes){
      if (is.null(saveDir)){
	 stop('No saveDir provided for saveRes. Please set saveDir.')
      }
      else {
	 saveDir = paste0(saveDir);
      }
      if (!dir.exists(saveDir)){
         message("Making new results directory at ",saveDir,"\n")
         dir.create(saveDir)
      }
   }
   
   message("Extracting Copy Number Features.\n")
   results[["CN_features"]] = extractCNFeats(segData,gbuild,cores,featsToUse)
   
   #If the component object is null, then perform the modeling
   if (is.null(components)){
     message("Fitting the component models.\n")
     results[["CN_components"]] = fitModels(results[["CN_features"]],cores=cores,pR = pR,featsToModel=featsToUse)
   }
   else {
     results[["CN_components"]] = components
   }
   
   # Creates the scm
   message("Generating sample-component-matrix.\n")
   results[["scm"]] = generateSCM(results[["CN_features"]],results[["CN_components"]])
   
   
   if (!is.null(ploidyData)){
      results[["scm"]] = addPloidyData(results[["scm"]], ploidyData)
   }
   
   # Plots the scm
   plotScm(results[["scm"]],runName,saveRes,saveDir)
   
   message("Determining # of Signatures via NMF.\n")
   determineNumSigs(results[["scm"]], rmin, rmax,cores,saveRes=saveRes,
                    saveDir = saveDir, runName = runName)
   
   return(results)

}
