#' createSigs
#'
#' This function is used to create the final signatures and generates the resulting
#' NMF object, from which you can extract the feature contribution to each
#' signature using NMF::basis(), and the signature contribution of each sample by
#' using NMF::scoef()
#' 
#' @param scm The sample_by_component matrix to run NMF on
#' @param nsig Number of signatures for the NMF to create
#' @param cores Number of cores to use in parallel process
#' @param runName Name of the run used in file names, Default is ""
#' @param saveRes Whether or not to save the results, Default is FALSE
#' @param saveDir Where to save the results, must be provided if using saveDir
#' @keywords signatures NMF
#' @return Returns the resulting NMF object
#' @export
#' @import NMF
#' @examples
#' \donttest{
#' createSigs(scmExp,5) #Generates 5 signatures from the SCM
#' }
createSigs = function(scm, nsig, cores = 1, runName = "",
                      saveRes=FALSE, saveDir=NULL) {

   sigs = NMF::nmf(t(scm),nsig,"KL",seed=77777,nrun=250,.opt = paste0("p",cores))
   
   if (saveRes){
      if (is.null(saveDir)){
      	stop('No saveDir provided for saveRes. Please set saveDir.')
      }
      #Save the Signature Exposure Matrix
      toWrite = t(NMF::scoef(sigs))
      colnames(toWrite) = paste0(rep("S",nsig),1:nsig)
      rownames(toWrite) = rownames(scm)
      utils::write.table(toWrite,sep = '\t',
                  file=paste0(saveDir,"/",runName,"_sigExposureMatrix.txt"))

      #Save the Feature Contribution matrix
      toWrite = NMF::basis(sigs)
      colnames(toWrite) = paste0(rep("S",nsig),1:nsig)
      utils::write.table(toWrite, sep = '\t',
                  file=paste0(saveDir,"/",runName,"_featureContribution.txt"))
   }
   
   return(sigs)
}
