\name{pett}
\alias{pett}
\title{
Nonparametric Change-Point Detection
}
\description{
Locates a single change-point in an annual series based on the Pettitt test.
}
\usage{
pett(x, plot = FALSE, order = FALSE, pval = .05,
  pchs = c(19, 21), ...)}

\arguments{
\item{x}{
a numeric vector, matrix or data frame with no missing
interior values
}
\item{plot}{
Should the trends be plotted when x is a matrix?
}
\item{order}{
Should the plotted trends be ordered by size?
}
\item{pval}{
p-value for significance
}
\item{pchs}{
Plot symbols for significant and not significant trend
estimates, respectively
}
\item{...}{
Other arguments to pass to plotting function
}
}

\details{
Pettitt's (1979) method is a rank-based nonparametric test for abrupt changes in a time series. It uses the Mann-Whitney statistic for testing that two samples (before and after the change-point) come from the same distribution, choosing the change-point that maximizes the statistic. The \emph{p}-value is approximate but accurate to 0.01 for \eqn{p \le} 0.5. Serial correlation is ignored, so the interval between points should be long enough to avoid strong serial correlation.

Missing values are allowed at the beginning or end of each variable but interior missing values will produce an NA. Otherwise the change-point might not be meaningful.

If \code{plot = TRUE}, a dot plot of \code{change.times} is shown. If \code{sort = TRUE}, the dots are sorted by \code{change.time}. The plot symbols indicate, respectively, that the trend is significant or not significant. The plot can be customized by passing any arguments used by \code{\link{dotchart}} such as \code{xlab}, as well as graphical parameters described in \code{\link{par}}.
}
\value{
A list of the following if \code{x} is a vector:
  \item{pettitt.K}{Pettitt's statistic}
  \item{p.value}{significance probability for statistic}
  \item{change.point}{last position preceding change to new level}
  \item{change.time}{if available, time of change.point position}
or a matrix with corresponding columns if \code{x} is a matrix or data frame.
}
\references{
Pettitt, A. N. (1979) A non-parametric approach to the change-point problem. \emph{Journal of the Royal Statistical Society. Series C (Applied Statistics)} \bold{28(2),} 126--135.
}
\note{
The \code{change.point} returned by these functions is the last position before the series actually changes, for consistency with the original Pettitt test. But for reporting purposes, the following position might be more appropriate to call the \dQuote{change-point}.

The Pettitt test produces a supposed change-point, even when the trend is smooth, or when the abrupt change is smaller than the long-term smooth change. Remove any smooth, long-term trend before applying this test.
}
\examples{
# data from Pettitt (1979, Table 1):
y <- c(-1.05, 0.96, 1.22, 0.58, -0.98, -0.03, -1.54, -0.71, -0.35, 0.66, 0.44,
  0.91, -0.02, -1.42, 1.26, -1.02, -0.81, 1.66, 1.05, 0.97, 2.14, 1.22, -0.24,
  1.60, 0.72, -0.12, 0.44, 0.03, 0.66, 0.56, 1.37, 1.66, 0.10, 0.80, 1.29, 0.49,
  -0.07, 1.18, 3.29, 1.84)
pett(y) # K=232, p=0.0146, change-point=17, the same results as Pettitt
# identify the year of a change-point in an annual time series:
pett(Nile)
# apply to a matrix time series:
y <- ts.intersect(Nile, LakeHuron)
pett(y)
pett(y, plot = TRUE, xlab = "Change-point")
legend("topleft", legend = "p < 0.05", pch = 19, bty="n")
# note how a smooth trend can disguise a change-point:
y <- 1:100 + c(rep(0, 75), rep(10, 25)) # smooth trend with change-point at 75
pett(y) # gives 50, erroneously
pett(residuals(lm(y~I(1:100)))) # removing trend gives 75, correctly
}
\keyword{ts}
\keyword{nonparametric}
