% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confidence_intervals.R
\name{confint_betabinom}
\alias{confint_betabinom}
\alias{confint_betabinom.wt_model}
\title{Beta Binomial Confidence Bounds for Quantiles and Probabilities}
\usage{
confint_betabinom(x, ...)

\method{confint_betabinom}{wt_model}(
  x,
  b_lives = c(0.01, 0.1, 0.5),
  bounds = c("two_sided", "lower", "upper"),
  conf_level = 0.95,
  direction = c("y", "x"),
  ...
)
}
\arguments{
\item{x}{Object with class \code{wt_model} and one of the classes
\code{wt_model_estimation} or \code{wt_model_estimation_list} returned from
\code{\link{rank_regression}}.}

\item{...}{Further arguments passed to or from other methods. Currently not used.}

\item{b_lives}{A numeric vector indicating the probabilities \emph{p} of the
\eqn{B_p}-lives (quantiles) to be considered.}

\item{bounds}{A character string specifying of which bounds have to be computed.
One of \code{"two_sided"}, \code{"lower"} or \code{"upper"}.}

\item{conf_level}{Confidence level of the interval.}

\item{direction}{A character string specifying the direction of the
confidence interval. One of \code{"y"} (failure probabilities) or \code{"x"}
(quantiles).}
}
\value{
A tibble with class \code{wt_confint} containing the following columns:
  \itemize{
    \item \code{x} : An ordered sequence of the lifetime characteristic regarding
      the failed units, starting at \code{min(x)} and ending up at \code{max(x)}.
      With \code{b_lives = c(0.01, 0.1, 0.5)} the 1\%, 10\% and 50\% quantiles
      are additionally included in \code{x}, but only if the specified
      probabilities are in the range of the estimated probabilities.
    \item \code{rank} : Interpolated ranks as a function of probabilities,
      computed with the converted approximation formula of Benard.
    \item \code{prob} : An ordered sequence of probabilities with specified
      \code{b_lives} included.
    \item \code{lower_bound} : Provided, if \code{bounds} is one of
      \code{"two_sided"} or \code{"lower"}. Lower limit of the confidence region
      with respect to \code{direction}, i.e. quantiles or probabilities.
    \item \code{upper_bound} : Provided, if \code{bounds} is one of
      \code{"two_sided"} or \code{"upper"}. Upper limit of the confidence region
      with respect to \code{direction}, i.e. quantiles or probabilities.
    \item \code{distribution} : Specified distribution (determined when calling
      \code{\link{rank_regression}}).
    \item \code{bounds} : Specified bound(s).
    \item \code{direction} : Specified direction.
    \item \code{cdf_estimation_method} : Specified method for the estimation
      of failure probabilities (determined when calling
      \code{\link{estimate_cdf}}).
  }
}
\description{
This non-parametric approach computes the beta binomial bounds (BB) for
quantiles and failure probabilities using a procedure similar to the calculation
of probabilities in terms of (\emph{Median Ranks}).
}
\details{
The difference to \emph{Median Ranks}, i.e. finding the probability of
rank \emph{j} at a 50\% level, is to determine the probability of rank \emph{j}
on another level, the specified confidence level.
}
\examples{
# Reliability data preparation:
## Data for two-parametric model:
data_2p <- reliability_data(
  shock,
  x = distance,
  status = status
)

## Data for three-parametric model:
data_3p <- reliability_data(
  alloy,
  x = cycles,
  status = status
)

# Probability estimation:
prob_tbl_2p <- estimate_cdf(
  data_2p,
  methods = "johnson"
)

prob_tbl_3p <- estimate_cdf(
  data_3p,
  methods = "johnson"
)

prob_tbl_mult <- estimate_cdf(
  data_3p,
  methods = c("johnson", "mr")
)

# Model estimation with rank_regression():
rr_2p <- rank_regression(
  prob_tbl_2p,
  distribution = "weibull"
)

rr_3p <- rank_regression(
  prob_tbl_3p,
  distribution = "lognormal3",
  conf_level = 0.90
)

rr_lists <- rank_regression(
  prob_tbl_mult,
  distribution = "loglogistic3",
  conf_level = 0.90
)

# Example 1 - Two-sided 95\% confidence interval for probabilities ('y'):
conf_betabin_1 <- confint_betabinom(
  x = rr_2p,
  bounds = "two_sided",
  conf_level = 0.95,
  direction = "y"
)

# Example 2 - One-sided lower/upper 90\% confidence interval for quantiles ('x'):
conf_betabin_2_1 <- confint_betabinom(
  x = rr_2p,
  bounds = "lower",
  conf_level = 0.90,
  direction = "x"
)

conf_betabin_2_2 <- confint_betabinom(
  x = rr_2p,
  bounds = "upper",
  conf_level = 0.90,
  direction = "x"
)

# Example 3 - Two-sided 90\% confidence intervals for both directions using
# a three-parametric model:
conf_betabin_3_1 <- confint_betabinom(
  x = rr_3p,
  bounds = "two_sided",
  conf_level = 0.90,
  direction = "y"
)

conf_betabin_3_2 <- confint_betabinom(
  x = rr_3p,
  bounds = "two_sided",
  conf_level = 0.90,
  direction = "x"
)

# Example 4 - Confidence intervals if multiple methods in estimate_cdf, i.e.
# "johnson" and "mr", were specified:

conf_betabin_4 <- confint_betabinom(
  x = rr_lists,
  bounds = "two_sided",
  conf_level = 0.99,
  direction = "y"
)

}
