% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/spheroid.R
\name{simSpheroidSystem}
\alias{simSpheroidSystem}
\title{Simulation of spheroid system}
\usage{
simSpheroidSystem(theta, size, shape = "const", orientation = "rbetaiso",
  stype = c("prolate", "oblate"), rjoint = NULL, box = list(c(0, 1)),
  pl = 0)
}
\arguments{
\item{theta}{simulation parameters}

\item{size}{name of random generating function for size distribution}

\item{shape}{scalar shape parameter}

\item{orientation}{name of random generating function for orientation distribution}

\item{stype}{spheroid type}

\item{rjoint}{name of joint random generating function}

\item{box}{simulation box}

\item{pl}{print level}
}
\value{
list of spheroids either of class \code{prolate} or \code{oblate}
}
\description{
Simulation of Poisson spheroid system
}
\details{
The function simulates a Poisson spheroid system according to the supplied
simulation parameter \code{theta} in a predefined simulation box.
For demonstration purposes there are three types
of major-axis length distributions available which are set by the \code{size} argument.

Further the function simulates either \code{stype}="prolate" or \code{stype}='oblate' spheroids.
For the directional orientation of the spheroid's major-axis one has the choice of a uniform
(\code{runifdir}), isotropic random planar (\code{rbetaiso}, see reference) or von Mises-Fisher (\code{rvMisesFisher})
distribution. The simulation box is a list containing of vector arguments which correspond to the lower and upper
points in each direction. If the argument \code{box} has only one element, i.e. \code{list(c(0,1)}, the same extent is
used for the other dimensions. If \code{rjoint} names a joint random generating function then argument \code{size} is ignored.

The argument \code{pl} denotes the print level of output information during simulation.
Currently, only \code{pl}=0 for no output and \code{pl}>100 for some additional info is implemented.
}
\examples{
\dontrun{
# directional distribution
rbetaiso <- function(kappa) {
   phi <- runif(1,0,1)*2*pi
   q <- runif(1,0,1)
   theta=acos((1-2*q)/sqrt(kappa*kappa-(1-2*q)*(1-2*q)*(kappa*kappa-1)))
   list("u"=c(sin(theta)*cos(phi),sin(theta)*sin(phi),cos(theta)),
		   "theta"=theta,"phi"=phi)					
}
   
# multivariate size distribution and orientation distribution 
rmulti <- function(m,s,kappa) {	
   dir <- rbetaiso(kappa)
   M <- chol(s, pivot = TRUE)
   M <- M[, order(attr(M, "pivot"))]
   x <- exp(matrix(m,nrow=1) +
          matrix(rnorm(ncol(s)), nrow = 1, byrow = TRUE) \%*\%M)
   a <- min(x)
   b <- max(x)
   
   list("a"=a,"b"=b,"u"=dir$u,"shape"=a/b,
        "theta"=dir$theta, "phi"=dir$phi)

}

set.seed(1234)
sigma <- matrix(c(0.1,0.1,0.1,0.25), ncol=2)
theta <- list("lam"=500,"rmulti"=list("m"=c(-3.0,-2.0),"s"=sigma,"kappa"=0.5))
S <- simSpheroidSystem(theta,rjoint="rmulti",box=list(c(0,5)),pl=101)

# Spheroids with lognormal distributed length of major axis
theta <- list("lam"=1000,"size"=list("meanlog"=-2.5,"sdlog"=0.5),
              "shape"=list("s"=0.5),
              "orientation"=list("kappa"=1.5))
S <- simSpheroidSystem(theta,size="rlnorm",orientation="rbetaiso",
                       box=list(c(0,5)),pl=101)

# Spheroids of constant sizes
theta <- list("lam"=1000,"size"=list(0.25),"shape"=list("s"=0.5),
              "orientation"=list("kappa"=1))
S <- simSpheroidSystem(theta,size="const",orientation="rbetaiso",
                       box=list(c(0,5)),pl=101)
}
}
\references{
Ohser, J. and Schladitz, K. 3D images of materials structures Wiley-VCH, 2009
}

