% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TARMAGARCH.test.R
\name{TARMAGARCH.test}
\alias{TARMAGARCH.test}
\title{ARMA GARCH versus TARMA GARCH supLM test for nonlinearity}
\usage{
TARMAGARCH.test(
  x,
  pa = 0.25,
  pb = 0.75,
  ar.ord = 1,
  ma.ord = 1,
  arch.ord = 1,
  garch.ord = 1,
  d = 1,
  thd.range,
  method = "CSS",
  ...
)
}
\arguments{
\item{x}{A univariate time series.}

\item{pa}{Real number in \code{[0,1]}. Sets the lower limit for the threshold search to the \code{100*pa}-th sample percentile.
The default is \code{0.25}}

\item{pb}{Real number in \code{[0,1]}. Sets the upper limit for the threshold search to the \code{100*pb}-th sample percentile.
The default is \code{0.75}}

\item{ar.ord}{Order of the AR part.}

\item{ma.ord}{Order of the MA part.}

\item{arch.ord}{Order of the ARCH part.}

\item{garch.ord}{Order of the GARCH part.}

\item{d}{Delay parameter. Defaults to \code{1}.}

\item{thd.range}{Vector of optional user defined threshold range. If missing then \code{pa} and \code{pb} are used.}

\item{method}{Fitting method to be passed to \code{arima}.}

\item{\dots}{Additional arguments to be passed to \code{arima}.}
}
\value{
A list of class \code{htest} with components:
\describe{
\item{\code{statistic}}{The value of the supLM statistic.}
\item{\code{parameter}}{A named vector: \code{threshold} is the value that maximises the Lagrange Multiplier values.}
\item{\code{test.v}}{Vector of values of the LM statistic for each threshold given in \code{thd.range}.}
\item{\code{thd.range}}{Range of values of the threshold.}
\item{\code{fit.ARMA}}{The null model: ARMA fit over \code{x}.}
\item{\code{fit.GARCH}}{The null model: GARCH fit over the residuals of the ARMA fit.}
\item{\code{sigma2}}{Estimated innovation variance from the ARMA fit.}
\item{\code{data.name}}{A character string giving the name of the data.}
\item{\code{prop}}{Proportion of values of the series that fall in the lower regime.}
\item{\code{p.value}}{The p-value of the test. It is \code{NULL} for the asymptotic test.}
\item{\code{method}}{A character string indicating the type of test performed.}
\item{\code{d}}{The delay parameter.}
\item{\code{pa}}{Lower threshold quantile.}
\item{\code{dfree}}{Effective degrees of freedom. It is the number of tested parameters.}
}
}
\description{
Implements a supremum Lagrange Multiplier test for a ARMA-GARCH specification versus
a TARMA-GARCH specification. Both the AR and MA parameters are tested
}
\details{
Implements an asymptotic supremum Lagrange Multiplier test to test an ARMA-GARCH specification versus
a TARMA-GARCH specification. In other words, the test is robust with respect to heteroskedasticity.
Both the AR parameters and the MA parameters are tested. This is an asymptotic test and the value of
the test statistic has to be compared with the critical values tabulated in \insertCite{Ang22}{tseriesTARMA}
or \insertCite{And03}{tseriesTARMA}.
}
\examples{
## Function to simulate from a ARMA-GARCH process

arma11.garch11 <- function(n, ph, th, a, b, a0=1, rand.gen = rnorm, innov = rand.gen(n, ...),
n.start = 500, start.innov = rand.gen(n.start, ...),...){

  #  Simulates a ARMA(1,1)-GARCH(1,1) process
  #  with parameeters ph, th, a, b, a0.
  #         x[t] <- ph*x[t-1] + th*eps[t-1] + eps[t]
  #       eps[t] <- e[t]*sqrt(v[t])
  #         v[t] <- a0 + a*eps[t-1]^2 + b*v[t-1];
  # ph  : AR
  # th  : MA
  # a   : ARCH
  # b   : GARCH

  # checks
  if(abs(a+b)>=1)   stop("model is not stationary")
  if(b/(1-a)>=1) stop("model has infinite fourth moments")

  if (!missing(start.innov) && length(start.innov) < n.start)
    stop(gettextf("'start.innov' is too short: need \%d points", n.start), domain = NA)
  e <- c(start.innov[1L:n.start], innov[1L:n])
  ntot <- length(e)
  x <- v <- eps <- double(ntot)
  v[1]   <- a0/(1.0-a-b);
  eps[1] <- e[1]*sqrt(v[1])
  x[1]   <- eps[1];
  for(i in 2:ntot){
    v[i]   <- a0 + a*eps[i-1]^2 + b*v[i-1];
    eps[i] <- e[i]*sqrt(v[i])
    x[i]   <- ph*x[i-1] + th*eps[i-1] + eps[i]
  }
  if (n.start > 0)  x <- x[-(1L:n.start)]
  return(ts(x));
}

## **************************************************************************
## Comparison between the robust and the non-robust test in presence of GARCH errors
## Simulates from the ARMA(1,1)-GARCH(1,1)

set.seed(12)
x1 <- arma11.garch11(n=100, ph=0.9, th=0.5, a=0.85, b=0.1, a0=1,n.start=500)
TARMAGARCH.test(x1, ar.ord=1, ma.ord=1, arch.ord=1, garch.ord=1, d=1)
TARMA.test(x1, ar.ord=1, ma.ord=1, d=1, ma.fixed=FALSE)

## a TARMA(1,1,1,1) where the threshold effect is on the AR parameters
set.seed(123)
x2  <- TARMA.sim(n=100, phi1=c(0.5,-0.5), phi2=c(0.0,0.8), theta1=0.5, theta2=0.5, d=1, thd=0.2)
TARMAGARCH.test(x2, ar.ord=1, ma.ord=1, d=1)


}
\references{
\itemize{
\item \insertRef{Ang22}{tseriesTARMA}
\item \insertRef{Gor21}{tseriesTARMA}
\item \insertRef{And03}{tseriesTARMA}
}
}
\seealso{
\code{\link{TARMA.test}} and \code{\link{TAR.test.B}} for the asymptotic and bootstrap test without the GARCH component. \code{\link{TARMA.sim}} to simulate from a TARMA process. \code{\link{TARMA.fit}} and \code{\link{TARMA.fit2}} for TARMA modelling.
}
\author{
Simone Giannerini, \email{simone.giannerini@unibo.it}

Greta Goracci, \email{greta.goracci@unibz.it}
}
