% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seqic_11.R
\name{seqic_indicator_11}
\alias{seqic_indicator_11}
\title{SEQIC Indicator 11 – Overtriage for Minor Trauma Patients}
\usage{
seqic_indicator_11(
  data,
  level,
  included_levels = c("I", "II", "III", "IV"),
  transfer_out_indicator,
  receiving_indicator,
  unique_incident_id,
  iss,
  ed_LOS,
  groups = NULL,
  calculate_ci = NULL,
  ...
)
}
\arguments{
\item{data}{A data frame containing trauma incident records.}

\item{level}{Column indicating the trauma center designation level (e.g., I,
II, III, IV).}

\item{included_levels}{Character vector indicating what facility levels to
include in the analysis.  Defaults to \code{c("I", "II", "III", "IV")}.}

\item{transfer_out_indicator}{Column name indicating whether the patient was
transferred out of the initial trauma center to definitive care. Logical,
character, or factor type. Values representing "No" (e.g., FALSE, "No")
indicate no transfer out.}

\item{receiving_indicator}{Column name indicating whether the
patient was transferred into the trauma center. Logical, character, or
factor type. Values representing "Yes" (e.g., TRUE, "Yes") indicate
transfer in.}

\item{unique_incident_id}{Unique identifier for each record.}

\item{iss}{Optional numeric column representing the Injury Severity Score.}

\item{ed_LOS}{Column for the calculated ED length of stay, measured in
minutes.}

\item{groups}{Additional columns passed as a vector of strings to
\code{dplyr::summarize()} via the \code{.by} argument for grouped summaries. Defaults
to \code{NULL}.}

\item{calculate_ci}{If \code{NULL}, 95\% confidence intervals will not be
calculated for the performance estimates.  Otherwise, options of "wilson"
or "clopper-pearson" can be supplied to utilize the corresponding methods
to calculate the confidence intervals for the proportions. Defaults to
\code{NULL}.}

\item{...}{
  Arguments passed on to \code{\link[nemsqar:nemsqa_binomial_confint]{nemsqar::nemsqa_binomial_confint}}
  \describe{
    \item{\code{conf.level}}{Numeric value between 0 and 1 indicating the confidence
level. Defaults to 0.95 (95\% confidence interval).}
    \item{\code{correct}}{Logical, indicating whether to apply continuity correction for
Wilson intervals. Defaults to \code{TRUE}.}
  }}
}
\value{
A tibble summarizing the numerator, denominator, and proportion of
overtriaged patients (Indicator 11), with optional 95\% confidence
intervals.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

Calculates SEQIC Indicator 11, which estimates the proportion of minor trauma
patients who were transferred into a trauma center and remained in the
Emergency Department for less than 24 hours. This indicator is designed to
identify potential overtriage events within the trauma system. Minor trauma
patients are identified using the Injury Severity Score (ISS < 9). Patients
must not have been transferred out and must have been received at a trauma
center level included in \code{included_levels}.
}
\details{
This function:
\itemize{
\item Filters the dataset to include only patients treated at trauma
centers designated Level I through IV.
\item Excludes patients transferred out and retains only those received by
the trauma center.
\item Deduplicates incident-level records using \code{unique_incident_id}.
\item Classifies patients as low-risk based on the  Injury Severity Score
(ISS < 9).
\item Flags low-risk patients who were discharged from the ED in under 24
hours.
\item Stratifies results by one or more user-defined grouping variables.
\item Returns a summarized tibble with the number of eligible low-risk
short-stay discharges (numerator), all received patients meeting inclusion
criteria (denominator), and the resulting proportion.
\item Optionally includes 95\% confidence intervals if \code{calculate_ci} is
specified.
}

Users must ensure appropriate column names are passed and data is
pre-processed to include the necessary fields without missing critical
identifiers or timestamps.
}
\examples{
# Packages
library(dplyr)
library(traumar)

# Simulated data for SEQIC Indicator 11
test_data <- tibble::tibble(
  id = as.character(1:10),
  trauma_level = c("I", "II", "III", "IV", "II", "I", "IV", "III", "II",
  "I"),
  transferred_out = c(FALSE, FALSE, FALSE, TRUE, FALSE, FALSE, TRUE, FALSE,
  FALSE, FALSE),
  received = c(TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE, TRUE),
  iss = c(4, 8, 10, 6, 5, 7, 6, 15, 3, 2),
  ed_LOS = c(6, 20, 30, 18, 8, 5, 22, 40, 2, 4),
  region = rep(c("East", "West"), each = 5)
)

# Run the function
traumar::seqic_indicator_11(
  data = test_data,
  level = trauma_level,
  included_levels = c("I", "II", "III", "IV"),
  transfer_out_indicator = transferred_out,
  receiving_indicator = received,
  unique_incident_id = id,
  iss = iss,
  ed_LOS = ed_LOS,
  groups = "region",
  calculate_ci = "clopper-pearson"
)

}
\references{
Roden-Foreman JW, Rapier NR, Yelverton L, Foreman ML. Asking a Better
Question: Development and Evaluation of the Need For Trauma Intervention
(NFTI) Metric as a Novel Indicator of Major Trauma. J Trauma Nurs. 2017
May/Jun;24(3):150-157. doi: 10.1097/JTN.0000000000000283. PMID: 28486318.
}
\author{
Nicolas Foss, Ed.D., MS
}
