% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nn-loss.R
\name{nn_bce_loss}
\alias{nn_bce_loss}
\title{Binary cross entropy loss}
\usage{
nn_bce_loss(weight = NULL, reduction = "mean")
}
\arguments{
\item{weight}{(Tensor, optional): a manual rescaling weight given to the loss
of each batch element. If given, has to be a Tensor of size \code{nbatch}.}

\item{reduction}{(string, optional): Specifies the reduction to apply to the output:
\code{'none'} | \code{'mean'} | \code{'sum'}. \code{'none'}: no reduction will be applied,
\code{'mean'}: the sum of the output will be divided by the number of
elements in the output, \code{'sum'}: the output will be summed. Note: \code{size_average}
and \code{reduce} are in the process of being deprecated, and in the meantime,
specifying either of those two args will override \code{reduction}. Default: \code{'mean'}}
}
\description{
Creates a criterion that measures the Binary Cross Entropy
between the target and the output:
}
\details{
The unreduced (i.e. with \code{reduction} set to \code{'none'}) loss can be described as:
\deqn{
  \ell(x, y) = L = \{l_1,\dots,l_N\}^\top, \quad
l_n = - w_n \left[ y_n \cdot \log x_n + (1 - y_n) \cdot \log (1 - x_n) \right]
}
where \eqn{N} is the batch size. If \code{reduction} is not \code{'none'}
(default \code{'mean'}), then

\deqn{
  \ell(x, y) = \left\{ \begin{array}{ll}
\mbox{mean}(L), & \mbox{if reduction} = \mbox{'mean';}\\
\mbox{sum}(L),  & \mbox{if reduction} = \mbox{'sum'.}
\end{array}
\right.
}

This is used for measuring the error of a reconstruction in for example
an auto-encoder. Note that the targets \eqn{y} should be numbers
between 0 and 1.

Notice that if \eqn{x_n} is either 0 or 1, one of the log terms would be
mathematically undefined in the above loss equation. PyTorch chooses to set
\eqn{\log (0) = -\infty}, since \eqn{\lim_{x\to 0} \log (x) = -\infty}.

However, an infinite term in the loss equation is not desirable for several reasons.
For one, if either \eqn{y_n = 0} or \eqn{(1 - y_n) = 0}, then we would be
multiplying 0 with infinity. Secondly, if we have an infinite loss value, then
we would also have an infinite term in our gradient, since
\eqn{\lim_{x\to 0} \frac{d}{dx} \log (x) = \infty}.

This would make BCELoss's backward method nonlinear with respect to \eqn{x_n},
and using it for things like linear regression would not be straight-forward.
Our solution is that BCELoss clamps its log function outputs to be greater than
or equal to -100. This way, we can always have a finite loss value and a linear
backward method.
}
\section{Shape}{

\itemize{
\item Input: \eqn{(N, *)} where \eqn{*} means, any number of additional
dimensions
\item Target: \eqn{(N, *)}, same shape as the input
\item Output: scalar. If \code{reduction} is \code{'none'}, then \eqn{(N, *)}, same
shape as input.
}
}

\examples{
if (torch_is_installed()) {
m <- nn_sigmoid()
loss <- nn_bce_loss()
input <- torch_randn(3, requires_grad=TRUE)
target <- torch_rand(3)
output <- loss(m(input), target)
output$backward()

}
}
