#' Specify the shape object
#' 
#' Creates a \code{\link{tmap-element}} that specifies the shape object. Also the projection and covered area (bounding box) can be set. It is possible to use multiple shape objects within one plot (see \code{\link{tmap-element}}).
#'  
#' @param shp shape object, which is one of
#' \enumerate{
#'  \item{\code{\link[sp:SpatialPolygonsDataFrame]{SpatialPolygons(DataFrame)}}}
#'  \item{\code{\link[sp:SpatialPointsDataFrame]{SpatialPoints(DataFrame)}}}
#'  \item{\code{\link[sp:SpatialLinesDataFrame]{SpatialLines(DataFrame)}}}
#'  \item{\code{\link[sp:SpatialGridDataFrame]{SpatialGrid(DataFrame)}}}
#'  \item{\code{\link[sp:SpatialPixelsDataFrame]{SpatialPixels(DataFrame)}}}
#'  \item{\code{\link[raster:Raster-class]{RasterLayer, RasterStack, or RasterBrick}}}
#' }
#' Simple features (\code{sf} objects) are also supported. For drawing layers \code{\link{tm_fill}} and \code{\link{tm_borders}}, 1 is required. For drawing layer \code{\link{tm_lines}}, 3 is required. Layers \code{\link{tm_symbols}} and \code{\link{tm_text}} accept 1 to 3. For layer \code{\link{tm_raster}}, 4, 5, or 6 is required.
#' @param name name of the shape object (character) as it appears in the legend in \code{"view"} mode. Default value is the name of \code{shp}.
#' @param is.master logical that determines whether this \code{tm_shape} is the master shape element. The bounding box, projection settings, and the unit specifications of the resulting thematic map are taken from the \code{tm_shape} element of the master shape object. By default, the first master shape element with a raster shape is the master, and if there are no raster shapes used, then the first \code{tm_shape} is the master shape element.
#' @param projection Either a \code{\link[sp:CRS]{CRS}} object or a character value. If it is a character, it can either be a \code{PROJ.4} character string or a shortcut. See \code{\link[tmaptools:get_proj4]{get_proj4}} for a list of shortcut values. By default, the projection is used that is defined in the \code{shp} object itself, which can be obtained with \code{\link[tmaptools:get_projection]{get_projection}}.
#' @param bbox bounding box. One of the following:
#' \itemize{
#' \item A bounding box (either 2 by 2 matrix or an \code{\link[raster:Extent]{Extent}} object). 
#' \item Open Street Map search query. The bounding is automatically generated by querying \code{q} from Open Street Map Nominatim. See \url{http://wiki.openstreetmap.org/wiki/Nominatim}.}
#' If unspecified, the current bounding box of \code{shp} is taken. The bounding box is feed to \code{\link[tmaptools:bb]{bb}} (as argument \code{x}. The other arguments of \code{\link[tmaptools:bb]{bb}} can be specified directly as well (see \code{..}). 
#' @param unit desired units of the map. One of \code{"metric"} (default), \code{"imperial"}, \code{"km"}, \code{"m"}, \code{"mi"} and \code{"ft"}. For other units, please specify \code{orig} and \code{to}, which are passed on to \code{\link[tmaptools:projection_units]{projection_units}}. Used to specify the scale bar (see \code{\link{tm_scale_bar}}) and to calculate densities for choropleths (see argument \code{convert2density} in \code{\link{tm_fill}}).
#' @param simplify simplification factor for spatial polygons and spatial lines. A number between 0 and 1 that indicates how many coordinates are kept. See the underlying function \code{\link[tmaptools:simplify_shape]{simplify_shape}}, from which the arguments \code{keep.units} and \code{keep.subunits} can be passed on (see \code{...}).
#' @param line.center.type vector of two values specifying how the center of spatial lines is determined Only applicable if \code{shp} is a \code{\link[sp:SpatialLinesDataFrame]{SpatialLines(DataFrame)}}, and symbols, dots, and/or text labels are used for this shape. The two values are:
#' \describe{
#' \item{\code{"feature", "single"}}{If \code{"feature"} is specified, a pair of coordinates (used for symbols, dots, and text labels) is chosen for each feature (i.e., a row in the \code{\link[sp:SpatialLinesDataFrame]{SpatialLines(DataFrame)}}). If \code{"segment"} is specified, a pair of coordinates is chosen for each line segment.}
#' \item{\code{"midpoint"} or \code{"centroid"}}{The midpoint is the middle point on the line, so the coordinates (used for symbols, dots, and text labels) correspond to the midpoints of the line segments. In case the first value is \code{"feature"}, then per feature, the midpoint of the line segment that is closest to the centroid is taken.}
#' }
#' @param ... Arguments passed on to \code{\link[tmaptools:bb]{bb}} (e.g. \code{ext} can be used to enlarge or shrinke a bounding box), \code{\link[tmaptools:projection_units]{projection_units}} (the arguments \code{orig} and \code{to}), and \code{\link[tmaptools:simplify_shape]{simplify_shape}} (the arguments \code{keep.units} and \code{keep.subunits})
#' @export
#' @seealso \code{\link[tmaptools:read_shape]{read_shape}} to read ESRI shape files, \code{\link[tmaptools:set_projection]{set_projection}}, \href{../doc/tmap-nutshell.html}{\code{vignette("tmap-nutshell")}} 
#' @references Tennekes, M., 2018, {tmap}: Thematic Maps in {R}, Journal of Statistical Software, 84(6), 1-39, \href{https://doi.org/10.18637/jss.v084.i06}{DOI}
#' @example ./examples/tm_shape.R
#' @return \code{\link{tmap-element}}
tm_shape <- function(shp, 
					 name = NULL,
					 is.master = NA,
					 projection=NULL,
					 bbox = NULL,
					 unit = getOption("tmap.unit"),
					 simplify = 1,
					 line.center.type = c("segment", "midpoint"),
					 ...) {
	shp_name <- ifelse(is.null(name) == TRUE, deparse(substitute(shp))[1], name)
	g <- list(tm_shape=c(as.list(environment()), list(...)))
	class(g) <- "tmap"
	g
}
