% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tq_portfolio.R
\name{tq_portfolio}
\alias{tq_portfolio}
\alias{tq_portfolio_}
\alias{tq_repeat_df}
\title{Aggregates a group of returns by asset into portfolio returns}
\usage{
tq_portfolio(
  data,
  assets_col,
  returns_col,
  weights = NULL,
  col_rename = NULL,
  ...
)

tq_portfolio_(
  data,
  assets_col,
  returns_col,
  weights = NULL,
  col_rename = NULL,
  ...
)

tq_repeat_df(data, n, index_col_name = "portfolio")
}
\arguments{
\item{data}{A \code{tibble} (tidy data frame) of returns in tidy format (i.e long format).}

\item{assets_col}{The column with assets (securities)}

\item{returns_col}{The column with returns}

\item{weights}{Optional parameter for the asset weights, which can be passed as a numeric vector the length of
the number of assets or a two column tibble with asset names in first column
and weights in second column.}

\item{col_rename}{A string or character vector containing names that can be used
to quickly rename columns.}

\item{...}{Additional parameters passed to \code{PerformanceAnalytics::Returns.portfolio}}

\item{n}{Number of times to repeat a data frame row-wise.}

\item{index_col_name}{A renaming function for the "index" column, used when repeating data frames.}
}
\value{
Returns data in the form of a \code{tibble} object.
}
\description{
Aggregates a group of returns by asset into portfolio returns
}
\details{
\code{tq_portfolio} is a wrapper for \code{PerformanceAnalytics::Returns.portfolio}.
The main advantage is the results are returned as a \code{tibble} and the
function can be used with the \code{tidyverse}.

\code{assets_col} and \code{returns_col} are columns within \code{data} that are used
to compute returns for a portfolio. The columns should be in "long" format (or "tidy" format)
meaning there is only one column containing all of the assets and one column containing
all of the return values (i.e. not in "wide" format with returns spread by asset).

\code{weights} are the weights to be applied to the asset returns.
Weights can be input in one of three options:
\itemize{
\item Single Portfolio: A numeric vector of weights that is the same length as unique number of assets.
The weights are applied in the order of the assets.
\item Single Portfolio: A two column tibble with assets in the first column and weights in the second column.
The advantage to this method is the weights are mapped to the assets and any unlisted
assets default to a weight of zero.
\item Multiple Portfolios: A three column tibble with portfolio index in the first
column, assets in the second column, and weights in the third column. The tibble
must be grouped by portfolio index.
}

\code{tq_repeat_df} is a simple function that repeats
a data frame \code{n} times row-wise (long-wise), and adds a new column for a portfolio index.
The function is used to assist in Multiple Portfolio analyses, and
is a useful precursor to \code{tq_portfolio}.
}
\examples{
# Load libraries
library(tidyquant)
library(dplyr)

# Use FANG data set
data(FANG)

# Get returns for individual stock components
monthly_returns_stocks <- FANG \%>\%
    group_by(symbol) \%>\%
    tq_transmute(adjusted, periodReturn, period = "monthly")

##### Portfolio Aggregation Methods #####

# Method 1: Use tq_portfolio with numeric vector of weights

weights <- c(0.50, 0.25, 0.25, 0)
tq_portfolio(data = monthly_returns_stocks,
             assets_col = symbol,
             returns_col = monthly.returns,
             weights = weights,
             col_rename = NULL,
             wealth.index = FALSE)

# Method 2: Use tq_portfolio with two column tibble and map weights

# Note that GOOG's weighting is zero in Method 1. In Method 2,
# GOOG is not added and same result is achieved.
weights_df <- tibble(symbol = c("FB", "AMZN", "NFLX"),
                     weights = c(0.50, 0.25, 0.25))
tq_portfolio(data = monthly_returns_stocks,
             assets_col = symbol,
             returns_col = monthly.returns,
             weights = weights_df,
             col_rename = NULL,
             wealth.index = FALSE)

# Method 3: Working with multiple portfolios

# 3A: Duplicate monthly_returns_stocks multiple times
mult_monthly_returns_stocks <- tq_repeat_df(monthly_returns_stocks, n = 4)

# 3B: Create weights table grouped by portfolio id
weights <- c(0.50, 0.25, 0.25, 0.00,
             0.00, 0.50, 0.25, 0.25,
             0.25, 0.00, 0.50, 0.25,
             0.25, 0.25, 0.00, 0.50)
stocks <- c("FB", "AMZN", "NFLX", "GOOG")
weights_table <- tibble(stocks) \%>\%
    tq_repeat_df(n = 4) \%>\%
    bind_cols(tibble(weights)) \%>\%
    group_by(portfolio)

# 3C: Scale to multiple portfolios
tq_portfolio(data = mult_monthly_returns_stocks,
             assets_col = symbol,
             returns_col = monthly.returns,
             weights = weights_table,
             col_rename = NULL,
             wealth.index = FALSE)
}
\seealso{
\itemize{
\item \code{\link[=tq_transmute]{tq_transmute()}} which can be used to get period returns.
\item \code{\link[PerformanceAnalytics:Return.portfolio]{PerformanceAnalytics::Return.portfolio()}} which is the underlying function
that specifies which parameters can be passed via \code{...}
}
}
