#' Deprecated functions in stplanr
#'
#' These functions are depreciated and will be removed:
#'
#' @name stplanr-deprecated
NULL

# Functions for aggregating two-way OD pairs into 'oneway' lines
#' Aggregate ods so they become non-directional
#'
#' For example, sum total travel in both directions.
#' @param x A data frame or SpatialLinesDataFrame, representing an OD matrix
#' @param attrib A vector of column numbers or names
#' for deciding which attribute(s) of class numeric to
#' aggregate
#' @param id1 Optional (it is assumed to be the first column)
#' text string referring to the name of the variable containing
#' the unique id of the origin
#' @param id2 Optional (it is assumed to be the second column)
#' text string referring to the name of the variable containing
#' the unique id of the destination
#' @param stplanr.key A key of unique OD pairs regardless of the order,
#' autogenerated by [od_id_order()]
#' @return `onewayid` outputs a data.frame with rows containing
#' results for the user-selected attribute values that have been aggregated.
#' @family lines
#' @export
onewayid <- function(x, attrib, id1 = names(x)[1], id2 = names(x)[2],
                     stplanr.key = od_id_order(x, id1, id2)) {
  .Deprecated(new = "od_oneway", package = "od",
              msg = "See updated od_oneway function in stplanr, and the od package")
  UseMethod("onewayid")
}

#' @name onewayid
#' @details
#' Flow data often contains movement in two directions: from point A to point B
#' and then from B to A. This can be problematic for transport planning, because
#' the magnitude of flow along a route can be masked by flows the other direction.
#' If only the largest flow in either direction is captured in an analysis, for
#' example, the true extent of travel will by heavily under-estimated for
#' OD pairs which have similar amounts of travel in both directions.
#' Flows in both direction are often represented by overlapping lines with
#' identical geometries (see [flowlines()]) which can be confusing
#' for users and are difficult to plot.
#' @examples
#' \donttest{
#' # this function is deprecated so examples are not expected to run
#' # keeping the example code in there for now for posterity
#' flow_oneway <- onewayid(flow, attrib = 3)
#' nrow(flow_oneway) < nrow(flow) # result has fewer rows
#' sum(flow$All) == sum(flow_oneway$All) # but the same total flow
#' # using names instead of index for attribute
#' onewayid(flow, attrib = "All")
#' # using many attributes to aggregate
#' attrib <- which(vapply(flow, is.numeric, TRUE))
#' flow_oneway <- onewayid(flow, attrib = attrib)
#' colSums(flow_oneway[attrib]) == colSums(flow[attrib]) # test if the colSums are equal
#' # Demonstrate the results from onewayid and onewaygeo are identical
#' flow_oneway_geo <- onewaygeo(flowlines, attrib = attrib)
#' plot(flow_oneway$All, flow_oneway_geo$All)
#' flow_oneway_sf <- onewayid(flowlines_sf, 3)
#' plot(flow_oneway_geo, lwd = flow_oneway_geo$All / mean(flow_oneway_geo$All))
#' plot(flow_oneway_sf$geometry, lwd = flow_oneway_sf$All / mean(flow_oneway_sf$All))
#' }
#' @export
onewayid.data.frame <- function(x, attrib, id1 = names(x)[1], id2 = names(x)[2],
                                stplanr.key = od_id_order(x, id1, id2)) {
  if (is.numeric(attrib)) {
    attrib_names <- names(x)[attrib]
  } else {
    attrib_names <- attrib
    attrib <- which(names(x) %in% attrib)
  }

  # separate geometry for sf objects
  is_sf <- is(x, "sf")
  if (is_sf) {
    x_sf <- sf::st_sf(stplanr.key[3], geometry = sf::st_geometry(x))
    x <- sf::st_drop_geometry(x)
  }

  x <- dplyr::bind_cols(x, stplanr.key)

  x_oneway_numeric <- dplyr::group_by(x, stplanr.key) %>%
    dplyr::summarise_at(attrib, sum)

  x_oneway_binary <- dplyr::mutate(x, is_two_way = duplicated(stplanr.key)) %>%
    dplyr::group_by(stplanr.key) %>%
    dplyr::summarise(is_two_way = dplyr::last(.data$is_two_way)) %>%
    dplyr::select(-stplanr.key)

  x_oneway_character <- x %>%
    dplyr::transmute(
      id1 = stringr::str_split(.data$stplanr.key, " ", simplify = TRUE)[, 1],
      id2 = stringr::str_split(.data$stplanr.key, " ", simplify = TRUE)[, 2],
      stplanr.key = .data$stplanr.key
    ) %>%
    dplyr::group_by(stplanr.key) %>%
    dplyr::summarise(id1 = dplyr::first(id1), id2 = dplyr::first(id2)) %>%
    dplyr::select(-stplanr.key)

  x_oneway <- dplyr::bind_cols(
    x_oneway_character,
    x_oneway_numeric,
    x_oneway_binary
  )

  if (is_sf) {
    x_sf <- x_sf[!duplicated(x_sf$stplanr.key), ]
    x_oneway <- sf::st_as_sf(dplyr::inner_join(x_oneway, x_sf))
    # class(x_oneway) # sf
  }

  x_oneway$stplanr.key <- NULL
  names(x_oneway)[1:2] <- c(id1, id2)

  return(x_oneway)
}

#' @name onewayid
#' @examples
#' # with spatial data
#' data(flowlines)
#' fo <- onewayid(flowlines, attrib = "All")
#' head(fo@data)
#' plot(fo)
#' sum(fo$All) == sum(flowlines$All)
#' # test results for one line
#' n <- 3
#' plot(fo[n, ], lwd = 20, add = TRUE)
#' f_over_n <- rgeos::gEquals(fo[n, ], flowlines["All"], byid = TRUE)
#' sum(flowlines$All[f_over_n]) == sum(fo$All[n]) # check aggregation worked
#' plot(flowlines[which(f_over_n)[1], ], add = TRUE, col = "white", lwd = 10)
#' plot(flowlines[which(f_over_n)[2], ], add = TRUE, lwd = 5)
#' @export
onewayid.SpatialLines <- function(x, attrib, id1 = names(x)[1], id2 = names(x)[2],
                                  stplanr.key = od_id_order(x, id1, id2)) {
  x_geom <- sp::SpatialLines(x@lines, proj4string = sp::CRS(proj4string(x)))
  x <- x@data

  x_oneway <- onewayid(x, id1, id2, attrib = attrib)
  x_oneway$stplanr.key <- od_id_order(x_oneway[c(id1, id2)])$stplanr.key

  if (length(x_geom) != nrow(x_oneway)) {
    id_old <- paste(x[[id1]], x[[id2]])
    sel <- id_old %in% x_oneway$stplanr.key
    x_geom <- x_geom[sel, ]
  }

  x_oneway <- sp::SpatialLinesDataFrame(sl = x_geom, data = x_oneway, match.ID = FALSE)

  return(x_oneway)
}

#' Import GTFS shapes and route data to SpatialLinesDataFrame.
#'
#' Takes a string with the file path of the zip file with the GTFS feed,
#' imports the shapes (geometry), route and agency data and returns a
#' SpatialLinesDataFrame for the GTFS feed.
#'
#' @param gtfszip String with the file path of the GTFS feed zip file
#' @export
#' @examples
#' f <- system.file("extdata", "beartransit-ca-us.zip", package = "stplanr")
#' # update file to latest version
#' # see https://code.google.com/p/googletransitdatafeed/wiki/PublicFeeds
#' u <- "http://data.trilliumtransit.com/gtfs/beartransit-ca-us/beartransit-ca-us.zip"
#' # download.file(u, f)
#' gtfs <- gtfs2sldf(gtfszip = f)
#' plot(gtfs, col = gtfs$route_long_name)
#' plot(gtfs[gtfs$route_long_name == "Central Campus", ])
#' \dontrun{
#' # An example of a larger gtfs feed
#' download.file(
#'   "http://www.yrt.ca/google/google_transit.zip",
#'   paste0(tempdir(), "/gtfsfeed.zip")
#' )
#' yrtgtfs <- gtfs2sldf(paste0(tempdir(), "/gtfsfeed.zip"))
#' sp::plot(yrtgtfs, col = paste0("#", yrtgtfs$route_color))
#' }
gtfs2sldf <- function(gtfszip = "") {
  .Deprecated(new = "read_gtfs", package = "gtfs2gps", msg = "Many packages read gtfs files")
  if (gtfszip == "") {
    stop("Zip file required")
  }
  if (file.exists(gtfszip) == FALSE) {
    stop("Specified zip file does not exist")
  }

  gtfsfiles <- unzip(gtfszip, exdir = tempdir())

  gtfstrips <-
    read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/trips.txt"))
  if (all(charToRaw(substr(colnames(gtfstrips)[1], 1, 3)) == c(as.raw(239), as.raw(46), as.raw(46)))) {
    gtfstrips <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/trips.txt"),
        fileEncoding = "UTF-8-BOM"
      )
    gtfsroutes <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/routes.txt"),
        fileEncoding = "UTF-8-BOM"
      )
    gtfsagency <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/agency.txt"),
        fileEncoding = "UTF-8-BOM"
      )
    gtfsshape <-
      read.csv(
        stringsAsFactors = TRUE,
        paste0(tempdir(), "/shapes.txt"),
        fileEncoding = "UTF-8-BOM"
      )
  }
  else {
    gtfsroutes <-
      read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/routes.txt"))
    gtfsagency <-
      read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/agency.txt"))
    gtfsshape <-
      read.csv(stringsAsFactors = TRUE, paste0(tempdir(), "/shapes.txt"))
  }

  if (nrow(gtfsshape) == 0) {
    stop("GTFS shapes.txt file is empty.")
  }

  unlink(gtfsfiles)

  gtfslines <- sp::SpatialLinesDataFrame((
    gtfsshape %>%
      dplyr::group_by_( ~ shape_id) %>%
      dplyr::arrange_( ~ shape_pt_sequence) %>%
      dplyr::do_(gtfsline = "sp::Lines(sp::Line(as.matrix(.[,c('shape_pt_lon','shape_pt_lat')])),unique(.$shape_id))") %>%
      dplyr::ungroup() %>%
      dplyr::do_(
        gtfsline = "sp::SpatialLines(.[[2]],
                                    proj4string = sp::CRS('+proj=longlat +ellps=WGS84 +datum=WGS84 +no_defs'))"
      )
  )[[1]][[1]],
  data = gtfstrips %>%
    dplyr::inner_join(gtfsroutes) %>%
    dplyr::distinct_(
      ~ route_id,
      ~ shape_id,
      ~ route_short_name,
      ~ route_long_name,
      ~ route_desc,
      ~ route_type,
      ~ route_color,
      ~ route_text_color,
      ~ agency_id
    ) %>%
    dplyr::select_(
      ~ route_id,
      ~ shape_id,
      ~ route_short_name,
      ~ route_long_name,
      ~ route_desc,
      ~ route_type,
      ~ route_color,
      ~ route_text_color,
      ~ agency_id
    ) %>%
    dplyr::inner_join(gtfsagency) %>%
    dplyr::do_("`rownames<-`(.,.$shape_id)")
  )
  rm(gtfstrips, gtfsshape, gtfsagency, gtfsroutes)
  return(gtfslines)
}

#' Function that estimates flow between points or zones using the radiation model
#'
#' This is an implementation of the radiation model proposed in a paper
#' by Simini et al. (2012).
#'
#' @param p A SpatialPoints dataframe, the first column of which contains a unique ID
#' @param pop_var A character string representing the variable that corresponds
#' to the population of the zone or point
#' @param proportion A number representing the proportion of the population who
#' commute (1, the default, means 100 percent of the population commute to work)
#' @references
#' Simini, F., Gonzalez, M.C., Maritan, A., Barabasi, A.L., 2012. A universal model for
#' mobility and migration patterns. Nature. doi:10.1038/nature10856
#' @family od
#' @export
#' @examples
#' \donttest{
#' # load some points data
#' data(cents)
#' # plot the points to check they make sense
#' plot(cents)
#' class(cents)
#' # Create test population to model flows
#' set.seed(2050)
#' cents$population <- runif(n = nrow(cents), min = 100, max = 1000)
#' # estimate
#' flowlines_radiation <- od_radiation(cents, pop_var = "population")
#' flowlines_radiation$flow
#' sum(flowlines_radiation$flow, na.rm = TRUE) # the total flow in the system
#' sum(cents$population) # the total inter-zonal flow
#' plot(flowlines_radiation, lwd = flowlines_radiation$flow / 100)
#' points(cents, cex = cents$population / 100)
#' }
od_radiation <- function(p, pop_var = "population", proportion = 1) {
  .Deprecated(msg = "See the od package")
  l <- points2flow(p)
  l$flow <- NA
  for (i in 1:nrow(p)) {
    for (j in 1:nrow(p)) {
      if (i == j) next()
      m <- p[[pop_var]][i]
      n <- p[[pop_var]][j]
      sel_flow <- which(l$O == p@data[i, 1] & l$D == p@data[j, 1])
      # create circle the radius of which is the distance between i and j centered on i
      radius <- gprojected(shp = l[sel_flow, ], fun = rgeos::gLength)
      s_circle <- geo_buffer(shp = p[i, ], width = radius)
      p@proj4string <- s_circle@proj4string
      ps <- p[-c(i, j), ][s_circle, ]
      s <- sum(ps[[pop_var]])
      l$flow[sel_flow] <-
        p[[pop_var]][i] * proportion * ((m * n) / ((m + s) * (m + n + s)))
    }
  }
  l
}

#' Plan a route with the graphhopper routing engine
#'
#' **Note: See https://github.com/crazycapivara/graphhopper-r for modern interface**
#'
#' Provides an R interface to the graphhopper routing engine,
#' an open source route planning service.
#'
#' The function returns a SpatialLinesDataFrame object.
#' See <https://github.com/graphhopper> for more information.
#'
#' @param vehicle A text string representing the vehicle.
#' Can be bike (default), car or foot. See <https://graphhopper.com/api/1/docs/supported-vehicle-profiles/> for further details.
#'
#' @details
#'
#' To test graphopper is working for you, try something like this, but with
#' your own API key:
#' To use this function you will need to obtain an API key from
#' <https://graphhopper.com/#directions-api>.
#' It is assumed that you have set your api key as a system environment
#' for security reasons (so you avoid typing the API key in your code).
#' Do this by adding the following to your .Renviron file (see `?.Renviron`
#' or the 'api-packages' vignette at <https://cran.r-project.org/package=httr>
#' for more on this):
#'
#' `GRAPHHOPPER='FALSE-Key-eccbf612-214e-437d-8b73-06bdf9e6877d'`.
#'
#' (Note: key not real, use your own key.)
#'
#' `obj <- jsonlite::fromJSON(url)`
#'
#' Where `url` is an example api request from
#'  <https://github.com/graphhopper/directions-api/blob/master/routing.md>.
#'
#' @inheritParams route_cyclestreets
#' @inheritParams od_coords
#' @family routes
#' @export
#' @seealso route_cyclestreet
#' @examples
#' \dontrun{
#' from <- c(-0.12, 51.5)
#' to <- c(-0.14, 51.5)
#' r1 <- route_graphhopper(from = from, to = to, silent = FALSE)
#' r2 <- route_graphhopper(from = from, to = to, silent = FALSE, vehicle = "foot")
#' r3 <- route_graphhopper(from = from, to = to, silent = FALSE, vehicle = "car")
#' plot(r1)
#' plot(r2, add = TRUE, col = "blue") # compare routes
#' plot(r3, add = TRUE, col = "red")
#' }
route_graphhopper <- function(from, to, l = NULL, vehicle = "bike",
                              silent = TRUE, pat = NULL,
                              base_url = "https://graphhopper.com") {

  .Deprecated(new = "gh_get_route", package = "graphhopper",
              msg = "See github.com/crazycapivara/graphhopper-r")

  # Convert character strings to lon/lat if needs be
  coords <- od_coords(from, to, l)

  if (is.null(pat)) {
    pat <- api_pat("graphhopper")
  }

  httrmsg <- httr::modify_url(
    base_url,
    path = "/api/1/route",
    query = list(
      point = paste0(coords[1, c("fy", "fx")], collapse = ","),
      point = paste0(coords[1, c("ty", "tx")], collapse = ","),
      vehicle = vehicle,
      locale = "en-US",
      debug = "true",
      points_encoded = "false",
      key = pat
    )
  )
  if (silent == FALSE) {
    print(paste0("The request sent was: ", httrmsg))
  }
  httrreq <- httr::GET(httrmsg)
  httr::stop_for_status(httrreq)

  if (grepl("application/json", httrreq$headers$`content-type`) == FALSE) {
    stop("Error: Graphhopper did not return a valid result")
  }

  txt <- httr::content(httrreq, as = "text", encoding = "UTF-8")
  if (txt == "") {
    stop("Error: Graphhopper did not return a valid result")
  }

  obj <- jsonlite::fromJSON(txt)

  if (is.element("message", names(obj))) {
    if (grepl("Wrong credentials", obj$message) == TRUE) {
      stop("Invalid API key")
    }
  }
  route <- sp::SpatialLines(list(sp::Lines(list(sp::Line(obj$paths$points[[2]][[1]][, 1:2])), ID = "1")))

  climb <- NA # to set elev variable up

  # get elevation data if it was a bike trip
  if (vehicle == "bike") {
    change_elev <- obj$path$descend + obj$paths$ascend
  } else {
    change_elev <- NA
  }

  # Attribute data for the route
  df <- data.frame(
    time = obj$paths$time / (1000 * 60),
    dist = obj$paths$distance,
    change_elev = change_elev
  )

  route <- sp::SpatialLinesDataFrame(route, df)
  sp::proj4string(route) <- sp::CRS("+init=epsg:4326")
  route
}

#' Aggregate OD data between polygon geometries
#'
#' @section Details:
#' Origin-destination ('OD') flow data is often provided
#' in the form of 1 line per flow with zone codes of origin and destination
#' centroids. This function aggregates OD flows between polygon geometries
#' allocating the original flows to larger zones based on area.
#' @inheritParams od2line
#' @param aggzones A SpatialPolygonsDataFrame containing the new
#' boundaries to aggregate to.
#' @param aggzone_points Points representing origins of OD flows
#' (typically population-weighted centroids)
#' @param cols A character vector containing the names of columns on which to
#' apply FUN. By default, all numeric columns are aggregated.
#' @param aggcols A character vector containing the names of columns in
#' aggzones to retain in the aggregated data.frame. By default,
#' only the first column is retained. These columns are renamed with a prefix
#' of "o_" and "d_".
#' @param FUN Function to use on aggregation. Default is sum.
#' @inheritParams sp_aggregate
#' @return data.frame containing the aggregated od flows.
#' @family od
#' @export
#' @examples
#' zones$quadrant <- c(1, 2, 1, 4, 5, 6, 7, 1)
#' aggzones <- rgeos::gUnaryUnion(zones, id = zones@data$quadrant)
#' aggzones <- sp::SpatialPolygonsDataFrame(aggzones, data.frame(region = c(1:6)), match.ID = FALSE)
#' sp::proj4string(aggzones) <- sp::proj4string(zones)
#' aggzones_sf <- sf::st_as_sf(aggzones)
#' aggzones_sf <- sf::st_set_crs(aggzones_sf, sf::st_crs(zones_sf))
#' od_agg <- od_aggregate(flow, zones_sf, aggzones_sf)
#' colSums(od_agg[3:9]) == colSums(flow[3:9])
#' od_sf_agg <- od2line(od_agg, aggzones_sf)
#' plot(flowlines, lwd = flowlines$Bicycle)
#' plot(od_sf_agg$geometry, lwd = od_sf_agg$Bicycle, add = TRUE, col = "red")
od_aggregate <- function(flow, zones, aggzones,
                         aggzone_points = NULL,
                         cols = FALSE,
                         aggcols = FALSE,
                         FUN = sum,
                         prop_by_area = ifelse(identical(FUN, mean) == FALSE, TRUE, FALSE),
                         digits = getOption("digits")) {
  UseMethod("od_aggregate", zones)
  .Deprecated(new = "od_aggregate", package = "od",
              msg = "See github.com/itsleeds/od")

}
#' @export
od_aggregate.sf <- function(flow, zones, aggzones,
                            aggzone_points = NULL,
                            cols = FALSE,
                            aggcols = FALSE,
                            FUN = sum,
                            prop_by_area = ifelse(identical(FUN, mean) == FALSE, TRUE, FALSE),
                            digits = getOption("digits")) {
  flow_first_col <- colnames(flow)[1]
  flow_second_col <- colnames(flow)[2]
  zonesfirstcol <- colnames(zones)[1]
  aggzonesfirstcol <- colnames(aggzones)[1]

  if (identical(cols, FALSE)) {
    col_ids <- sapply(flow, is.numeric)
    cols <- names(col_ids)[col_ids]
  }
  if (aggcols == FALSE) {
    aggcols <- colnames(aggzones)[1]
  }

  zone_points <- sf::st_centroid(zones)
  if (is.null(aggzone_points)) {
    aggzone_points <- sf::st_centroid(aggzones)
  }

  zones_agg <- zone_points %>%
    sf::st_join(y = aggzones[aggcols]) %>%
    sf::st_set_geometry(NULL)

  names(zones_agg)[1] <- flow_first_col
  zones_agg$new_orig <- zones_agg[, aggcols[1]]
  zones_agg$new_dest <- zones_agg[, aggcols[1]]

  flow_new_orig <- flow %>%
    dplyr::inner_join(y = zones_agg[c(flow_first_col, "new_orig")])

  names(zones_agg)[1] <- flow_second_col

  flow_new_dest <- flow_new_orig %>%
    dplyr::inner_join(y = zones_agg[c(flow_second_col, "new_dest")])

  flow_ag <- flow_new_dest %>%
    dplyr::group_by(!!rlang::sym("new_orig"), !!rlang::sym("new_dest")) %>%
    dplyr::summarise_at(.vars = cols, .funs = sum) %>%
    dplyr::ungroup()

  flow_ag

  # od2line(flow = flow_ag, zones = aggzones) # to export as sf
}
#' @export
od_aggregate.Spatial <- function(flow, zones, aggzones,
                                 aggzone_points = NULL,
                                 cols = FALSE,
                                 aggcols = FALSE,
                                 FUN = sum,
                                 prop_by_area = ifelse(identical(FUN, mean) == FALSE, TRUE, FALSE),
                                 digits = getOption("digits")) {
  zonesfirstcol <- colnames(zones@data)[1]
  aggzonesfirstcol <- colnames(aggzones@data)[1]

  if (cols == FALSE) {
    cols <- unlist(lapply(flow, is, "numeric"))
    cols <- names(cols[which(cols == TRUE)])
  }
  if (aggcols == FALSE) {
    aggcols <- colnames(aggzones@data)[1]
  }

  origzones <- zones
  origaggzones <- aggzones

  if (sp::is.projected(zones) == TRUE & all.equal(zones@proj4string, aggzones@proj4string) == FALSE) {
    aggzones <- sp::spTransform(aggzones, zones@proj4string)
  } else {
    projection <- paste0(
      "+proj=aea +lat_1=90 +lat_2=-18.416667 ",
      "+lat_0=0 +lon_0=10 +x_0=0 +y_0=0 +ellps=GRS80",
      " +towgs84=0,0,0,0,0,0,0 +units=m +no_defs"
    )
    zones <- sp::spTransform(zones, projection)
    aggzones <- sp::spTransform(aggzones, projection)
  }

  zones@data$stplanr_area <- rgeos::gArea(zones, byid = TRUE)
  zones@data$od_aggregate_charid <- row.names(zones@data)
  aggzones@data$od_aggregate_charid <- row.names(aggzones@data)

  zoneintersect <- rgeos::gIntersection(zones, aggzones, byid = TRUE)
  zoneintersect <- sp::SpatialPolygonsDataFrame(zoneintersect,
                                                data = data.frame(
                                                  od_aggregate_charid = sapply(zoneintersect@polygons, function(x) x@ID),
                                                  row.names = sapply(zoneintersect@polygons, function(x) x@ID)
                                                )
  )
  zoneintersect@data$od_aggregate_interarea <- rgeos::gArea(zoneintersect, byid = TRUE)
  zoneintersect@data$od_aggregate_zone_charid <- stringr::str_split(zoneintersect@data$od_aggregate_charid, " ", simplify = TRUE)[, 1]
  zoneintersect@data$od_aggregate_aggzone_charid <- stringr::str_split(zoneintersect@data$od_aggregate_charid, " ", simplify = TRUE)[, 2]

  zoneintersect <- merge(zoneintersect, zones@data, by.x = "od_aggregate_zone_charid", by.y = "od_aggregate_charid")
  zoneintersect@data$od_aggregate_proparea <- zoneintersect@data$od_aggregate_interarea / zoneintersect@data$stplanr_area

  intersectdf <- merge(merge(
    flow,
    setNames(zoneintersect@data, paste0("o_", colnames(zoneintersect@data))),
    by.x = colnames(flow)[1],
    by.y = paste0("o_", zonesfirstcol)
  ),
  setNames(zoneintersect@data, paste0("d_", colnames(zoneintersect@data))),
  by.x = colnames(flow)[2],
  by.y = paste0("d_", zonesfirstcol)
  )

  if (prop_by_area == TRUE & is(zones, "SpatialPolygonsDataFrame") == TRUE) {
    intersectdf <- intersectdf %>%
      dplyr::mutate_at(
        cols, dplyr::funs_("round(.*o_od_aggregate_proparea*d_od_aggregate_proparea)", args = list("digits" = digits))
      )
  }

  intersectdf <- intersectdf %>%
    dplyr::group_by_("o_od_aggregate_aggzone_charid", "d_od_aggregate_aggzone_charid") %>%
    dplyr::select(dplyr::one_of(c("o_od_aggregate_aggzone_charid", "d_od_aggregate_aggzone_charid", cols))) %>%
    dplyr::summarise_at(cols, .funs = FUN) %>%
    dplyr::left_join(setNames(aggzones@data[, c("od_aggregate_charid", aggcols)], c("od_aggregate_charid", paste0("o_", aggcols))),
                     by = c("o_od_aggregate_aggzone_charid" = "od_aggregate_charid")
    ) %>%
    dplyr::left_join(setNames(aggzones@data[, c("od_aggregate_charid", aggcols)], c("od_aggregate_charid", paste0("d_", aggcols))),
                     by = c("d_od_aggregate_aggzone_charid" = "od_aggregate_charid")
    )
  intersectdf <- intersectdf[, c(
    paste0("o_", c(aggzonesfirstcol, aggcols[which(aggcols != aggzonesfirstcol)])),
    paste0("d_", c(aggzonesfirstcol, aggcols[which(aggcols != aggzonesfirstcol)])),
    cols
  )]

  return(as.data.frame(intersectdf))
}

#' Aggregate SpatialPolygonsDataFrame to new geometry.
#'
#' @section Details:
#' This function performs aggregation on a SpatialPolygonsDataFrame to a
#' different geometry specified by another SpatialPolygons object.
#' @inheritParams od2line
#' @param aggzones A SpatialPolygonsDataFrame containing the new
#' boundaries to aggregate to.
#' @param cols A character vector containing the names of columns on which to
#' apply FUN. By default, all numeric columns are aggregated.
#' @param FUN Function to use on aggregation. Default is sum.
#' @param prop_by_area Boolean value indicating if the values should be
#' proportionally adjusted based on area. Default is TRUE unless FUN = mean.
#' @param digits The number of digits to use when proportionally adjusting
#' values based on area. Default is the value of getOption("digits").
#'
#' @return SpatialPolygonsDataFrame
#' @family od
#'
#' @export
#' @examples
#' \dontrun{
#' zones@data$region <- 1
#' zones@data[c(2, 5), c("region")] <- 2
#' aggzones <- sp::SpatialPolygonsDataFrame(rgeos::gUnaryUnion(
#'   zones,
#'   id = zones@data$region
#' ), data.frame(region = c(1, 2)))
#' zones@data$region <- NULL
#' zones@data$exdata <- 5
#' library(sp)
#' sp_aggregate(zones, aggzones)
#' }
sp_aggregate <- function(zones, aggzones, cols = FALSE,
                         FUN = sum,
                         prop_by_area = ifelse(identical(FUN, mean) == FALSE, TRUE, FALSE),
                         digits = getOption("digits")) {
  .Deprecated(new = "od_aggregate", package = "od",
              msg = "See github.com/itsleeds/od")
  zonesfirstcol <- colnames(zones@data)[1]
  aggzonesfirstcol <- colnames(aggzones@data)[1]
  aggcols <- colnames(aggzones@data)

  if (cols == FALSE) {
    cols <- unlist(lapply(zones@data, is, "numeric"))
    cols <- names(cols[which(cols == TRUE)])
    cols <- cols[which(cols != zonesfirstcol)]
  }

  origzones <- zones
  origaggzones <- aggzones

  if (sp::is.projected(zones) == TRUE & all.equal(zones@proj4string, aggzones@proj4string) == FALSE) {
    aggzones <- sp::spTransform(aggzones, zones@proj4string)
  } else {
    projection <- paste0(
      "+proj=aea +lat_1=90 +lat_2=-18.416667 ",
      "+lat_0=0 +lon_0=10 +x_0=0 +y_0=0 +ellps=GRS80",
      " +towgs84=0,0,0,0,0,0,0 +units=m +no_defs"
    )
    zones <- sp::spTransform(zones, projection)
    aggzones <- sp::spTransform(aggzones, projection)
  }

  zones@data$stplanr_area <- rgeos::gArea(zones, byid = TRUE)
  zones@data$od_aggregate_charid <- row.names(zones@data)
  aggzones@data$od_aggregate_charid <- row.names(aggzones@data)

  zoneintersect <- rgeos::gIntersection(zones, aggzones, byid = TRUE)
  zoneintersect <- sp::SpatialPolygonsDataFrame(zoneintersect,
                                                data = data.frame(
                                                  od_aggregate_charid = sapply(zoneintersect@polygons, function(x) x@ID),
                                                  row.names = sapply(zoneintersect@polygons, function(x) x@ID)
                                                )
  )
  zoneintersect@data$od_aggregate_interarea <- rgeos::gArea(zoneintersect, byid = TRUE)
  zoneintersect@data$od_aggregate_zone_charid <- stringr::str_split(zoneintersect@data$od_aggregate_charid, " ", simplify = TRUE)[, 1]
  zoneintersect@data$od_aggregate_aggzone_charid <- stringr::str_split(zoneintersect@data$od_aggregate_charid, " ", simplify = TRUE)[, 2]

  zoneintersect <- merge(zoneintersect, zones@data, by.x = "od_aggregate_zone_charid", by.y = "od_aggregate_charid")
  zoneintersect@data$od_aggregate_proparea <- zoneintersect@data$od_aggregate_interarea / zoneintersect@data$stplanr_area

  intersectdf <- zoneintersect@data

  if (prop_by_area == TRUE & is(zones, "SpatialPolygonsDataFrame") == TRUE) {
    intersectdf <- intersectdf %>%
      dplyr::mutate_at(
        cols, dplyr::funs_("round(.*od_aggregate_proparea)", args = list("digits" = digits))
      )
  }

  intersectdf <- intersectdf %>%
    dplyr::group_by_("od_aggregate_aggzone_charid") %>%
    dplyr::select(dplyr::one_of(c("od_aggregate_aggzone_charid", cols))) %>%
    dplyr::summarise_at(cols, .funs = FUN) %>%
    dplyr::left_join(setNames(aggzones@data[, c("od_aggregate_charid", aggcols)], c("od_aggregate_aggzone_charid", aggcols)),
                     by = "od_aggregate_aggzone_charid"
    )
  intersectdf <- as.data.frame(
    intersectdf,
    intersectdf$od_aggregate_aggzone_charid
  )
  intersectdf <- intersectdf[, c(
    c(aggzonesfirstcol, aggcols[which(aggcols != aggzonesfirstcol)]),
    cols
  )]

  aggzones <- origaggzones
  aggzones@data <- intersectdf

  return(aggzones)
}


