% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prop_allocation.R
\name{prop_allocation}
\alias{prop_allocation}
\title{Construct a proportional allocation}
\usage{
prop_allocation(
  x,
  n,
  strata,
  initial = 0L,
  divisor = function(a) a + 1,
  ties = c("largest", "first")
)
}
\arguments{
\item{x}{A positive and finite numeric vector of sizes for units in the
population (e.g., revenue for drawing a sample of businesses).}

\item{n}{A positive integer giving the sample size.}

\item{strata}{A factor, or something that can be coerced into one, giving
the strata associated with units in the population. The default is to place
all units into a single stratum.}

\item{initial}{A positive integer vector giving the initial (or minimal)
allocation for each stratum, ordered according to the levels of
\code{strata}. A single integer is recycled for each stratum using a special
algorithm to ensure a feasible allocation; see details. Non-integers are
truncated towards 0. The default allows for no units to be allocated to a
stratum.}

\item{divisor}{A divisor function for the divisor (highest-averages)
apportionment method. The default uses the Jefferson (D'Hondt) method. See
details for other possible functions.}

\item{ties}{Either 'largest' to break ties in favor of the stratum with the
largest size, or 'first' to break ties in favor of the ordering of
\code{strata}.}
}
\value{
A named integer vector of sample sizes for each stratum in \code{strata}.
}
\description{
Generate a proportional-to-size allocation for stratified sampling.
}
\details{
The \code{prop_allocation()} function gives a sample size for each level in
\code{strata} that is proportional to the sum of \code{x} across strata and
adds up to \code{n}. This is done using the divisor (highest-averages)
apportionment method (Balinksi and Young, 1982, Appendix A), for which there
are a number of different divisor functions:

\describe{
\item{Jefferson/D'Hondt}{\verb{\\(a) a + 1}}
\item{Webster/Sainte-Laguë}{\verb{\\(a) a + 0.5}}
\item{Imperiali}{\verb{\\(a) a + 2}}
\item{Huntington-Hill}{\verb{\\(a) sqrt(a * (a + 1))}}
\item{Danish}{\verb{\\(a) a + 1 / 3}}
\item{Adams}{\verb{\\(a) a}}
\item{Dean}{\verb{\\(a) a * (a + 1) / (a + 0.5)}}
}

Note that a divisor function with \eqn{d(0) = 0} (i.e., Huntington-Hill,
Adams, Dean) should have an initial allocation of at least 1 for all strata.
In all cases, ties are broken according to the sum of \code{x} if
\code{ties = 'largest'}; otherwise, if \code{ties = 'first'}, then ties are broken
according to the levels of \code{strata}.

In cases where the number of units with non-zero size in a stratum is
smaller than its allocation, the allocation for that stratum is set to the
number of available units, with the remaining sample size reallocated to
other strata proportional to \code{x}. This is similar to \command{PROC
SURVEYSELECT} in SAS with \command{ALLOC = PROPORTIONAL}.

Passing a single integer for the initial allocation first checks that
recycling this value for each stratum does not result in an allocation
larger than the sample size. If it does, then the value is reduced so that
recycling does not exceed the sample size. This recycled vector can be
further reduced in cases where it exceeds the number of units in a stratum,
the result of which is the initial allocation. This special recycling
ensures that the initial allocation is feasible.
}
\examples{
# Make a population with units of different size
x <- c(rep(1:9, each = 3), 100, 100, 100)

# ... and 10 strata
s <- rep(letters[1:10], each = 3)

# Generate an allocation
prop_allocation(x, 15, s, initial = 1)

}
\references{
Balinksi, M. L. and Young, H. P. (1982).
\emph{Fair Representation: Meeting the Ideal of One Man, One Vote}.
Yale University Press.
}
\seealso{
\code{\link[=sps]{sps()}} for stratified sequential Poisson sampling.

\code{\link[=expected_coverage]{expected_coverage()}} to calculate the expected number of strata in a sample
without stratification.

\code{strAlloc()} in the \pkg{PracTools} package for other allocation methods.
}
