% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loudness.R
\name{getLoudness}
\alias{getLoudness}
\title{Get loudness}
\usage{
getLoudness(x, samplingRate = NULL, scale = NULL, windowLength = 50,
  step = NULL, overlap = 50, SPL_measured = 70, Pref = 2e-05,
  spreadSpectrum = TRUE, plot = TRUE, mar = c(5.1, 4.1, 4.1, 4.1),
  ...)
}
\arguments{
\item{x}{path to a .wav or .mp3 file or a vector of amplitudes with specified
samplingRate}

\item{samplingRate}{sampling rate of \code{x} (only needed if
\code{x} is a numeric vector, rather than an audio file), must be > 2000 Hz}

\item{scale}{the maximum possible value of \code{x} (only needed if \code{x}
is a numeric vector, rather than an audio file); defaults to observed
\code{max(abs(x))} if it is greater than 1 and to 1 otherwise}

\item{windowLength}{length of FFT window, ms}

\item{step}{you can override \code{overlap} by specifying FFT step, ms}

\item{overlap}{overlap between successive FFT frames, \%}

\item{SPL_measured}{sound pressure level at which the sound is presented, dB}

\item{Pref}{reference pressure, Pa}

\item{spreadSpectrum}{if TRUE, applies a spreading function to account for
frequency masking}

\item{plot}{should a spectrogram be plotted? TRUE / FALSE}

\item{mar}{margins of the spectrogram}

\item{...}{other plotting parameters passed to \code{\link{spectrogram}}}
}
\value{
Returns a list of length two: \describe{
  \item{specSone}{spectrum in sone: a matrix with frequency on the bark
  scale in rows and time (STFT frames) in columns}
  \item{loudness}{a vector of loudness per STFT frame (sone)}
  }
}
\description{
Estimates subjective loudness per frame, in sone. Based on EMBSD speech
quality measure, particularly the matlab code in Yang (1999) and Timoney et
al. (2004). Note that there are many ways to estimate loudness and many other
factors, ignored by this model, that could influence subjectively experienced
loudness. Please treat the output with a healthy dose of skepticism! Also
note that the absolute value of calculated loudness critically depends on the
chosen "measured" sound pressure level (SPL). \code{getLoudness} estimates
how loud a sound will be experienced if it is played back at an SPL of
\code{SPL_measured} dB. The most meaningful way to use the output is to
compare the loudness of several sounds analyzed with identical settings or of
different segments within the same recording.
}
\details{
Algorithm: calibrates the sound to the desired SPL (Timoney et al., 2004),
extracts a \code{\link{spectrogram}}, converts to bark scale
(\code{\link[tuneR]{audspec}}), spreads the spectrum to account for frequency
masking across the critical bands (Yang, 1999), converts dB to phon by using
standard equal loudness curves (ISO 226), converts phon to sone (Timoney et
al., 2004), sums across all critical bands, and applies a correction
coefficient to standardize output. Calibrated so as to return a loudness of 1
sone for a 1 kHz pure tone with SPL of 40 dB.
}
\examples{
sounds = list(
  white_noise = runif(8000, -1, 1),
  white_noise2 = runif(8000, -1, 1) / 2,  # ~6 dB quieter
  pure_tone_1KHz = sin(2*pi*1000/16000*(1:8000))  # pure tone at 1 kHz
)
loud = rep(0, length(sounds)); names(loud) = names(sounds)
for (i in 1:length(sounds)) {
  # playme(sounds[[i]], 16000)
  l = getLoudness(
    x = sounds[[i]], samplingRate = 16000, scale = 1,
    windowLength = 20, step = NULL,
    overlap = 50, SPL_measured = 40,
    Pref = 2e-5, plot = FALSE)
  loud[i] = mean(l$loudness)
}
loud
# white noise (sound 1) is twice as loud as pure tone at 1 KHz (sound 3),
# and note that the same white noise with lower amplitude has lower loudness
# (provided that "scale" is specified)
# compare: lapply(sounds, range)

\dontrun{
  s = soundgen()
  l = getLoudness(s, SPL_measured = 70,
                  samplingRate = 16000, plot = TRUE, osc = TRUE)
  # The estimated loudness in sone depends on target SPL
  l = getLoudness(s, SPL_measured = 40,
                  samplingRate = 16000, plot = TRUE)

  # ...but not (much) on windowLength and samplingRate
  l = getLoudness(soundgen(), SPL_measured = 40, windowLength = 50,
                  samplingRate = 16000, plot = TRUE)

  # input can be an audio file
  getLoudness('~/Downloads/temp/032_ut_anger_30-m-roar-curse.wav')
}
}
\references{
\itemize{
  \item ISO 226 as implemented by Jeff Tackett (2005) on
  https://www.mathworks.com/matlabcentral/fileexchange/
  7028-iso-226-equal-loudness-level-contour-signal \item Timoney, J.,
  Lysaght, T., Schoenwiesner, M., & MacManus, L. (2004). Implementing
  loudness models in matlab. \item Yang, W. (1999). Enhanced Modified Bark
  Spectral Distortion (EMBSD): An Objective Speech Quality Measure Based on
  Audible Distortion and Cognitive Model. Temple University. }
}
\seealso{
\code{\link{getLoudnessFolder}} \code{\link{getRMS}}
  \code{\link{analyze}}
}
