% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/source.R
\name{generateNoise}
\alias{generateNoise}
\title{Generate noise}
\usage{
generateNoise(len, rolloffNoise = 0, noiseFlatSpec = 1200,
  spectralEnvelope = NULL, noise = NULL, temperature = 0.1,
  attackLen = 10, windowLength_points = 1024, samplingRate = 16000,
  overlap = 75, dynamicRange = 80, play = FALSE)
}
\arguments{
\item{len}{length of output}

\item{rolloffNoise}{linear rolloff of the excitation source for the unvoiced
component, dB/kHz (anchor format)}

\item{noiseFlatSpec}{keeps noise spectrum flat to this frequency, Hz}

\item{spectralEnvelope}{(optional): as an alternative to using rolloffNoise,
we can provide the exact filter - a vector of non-negative numbers
specifying the power in each frequency bin on a linear scale (interpolated
to length equal to windowLength_points/2). A matrix specifying the filter
for each STFT step is also accepted. The easiest way to create this matrix
is to call soundgen:::getSpectralEnvelope or to use the spectrum of a
recorded sound}

\item{noise}{loudness of turbulent noise (0 dB = as loud as
voiced component, negative values = quieter) such as aspiration, hissing,
etc (anchor format)}

\item{temperature}{hyperparameter for regulating the amount of stochasticity
in sound generation}

\item{attackLen}{duration of fade-in / fade-out at each end of syllables and
noise (ms): a vector of length 1 (symmetric) or 2 (separately for fade-in
and fade-out)}

\item{windowLength_points}{the length of fft window, points}

\item{samplingRate}{sampling frequency, Hz}

\item{overlap}{FFT window overlap, \%. For allowed values, see
\code{\link[seewave]{istft}}}

\item{dynamicRange}{dynamic range, dB. Harmonics and noise more than
dynamicRange under maximum amplitude are discarded to save computational
resources}

\item{play}{if TRUE, plays the synthesized sound using the default player on
your system. If character, passed to \code{\link[tuneR]{play}} as the name
of player to use, eg "aplay", "play", "vlc", etc. In case of errors, try
setting another default player for \code{\link[tuneR]{play}}}
}
\description{
Generates noise of length \code{len} and with spectrum defined by linear
decay of \code{rolloffNoise} dB/kHz above \code{noiseFlatSpec} Hz OR by a
specified filter \code{spectralEnvelope}. This function is called internally
by \code{\link{soundgen}}, but it may be more convenient to call it directly
when synthesizing non-biological noises defined by specific spectral and
amplitude envelopes rather than formants: the wind, whistles, impact noises,
etc. See \code{\link{fart}} and \code{\link{beat}} for similarly simplified
functions for tonal non-biological sounds.
}
\details{
Algorithm: paints a spectrogram with desired characteristics, sets phase to
zero, and generates a time sequence via inverse FFT.
}
\examples{
# .5 s of white noise
samplingRate = 16000
noise1 = generateNoise(len = samplingRate * .5,
  samplingRate = samplingRate)
# playme(noise1, samplingRate)
# seewave::meanspec(noise1, f = samplingRate)

# Percussion (run a few times to notice stochasticity due to temperature = .25)
noise2 = generateNoise(len = samplingRate * .15, noise = c(0, -80),
  rolloffNoise = c(4, -6), attackLen = 5, temperature = .25)
noise3 = generateNoise(len = samplingRate * .25, noise = c(0, -40),
  rolloffNoise = c(4, -20), attackLen = 5, temperature = .25)
# playme(c(noise2, noise3), samplingRate)

\dontrun{
playback = c(TRUE, FALSE, 'aplay', 'vlc')[2]
# 1.2 s of noise with rolloff changing from 0 to -12 dB above 2 kHz
noise = generateNoise(len = samplingRate * 1.2,
  rolloffNoise = c(0, -12), noiseFlatSpec = 2000,
  samplingRate = samplingRate, play = playback)
# spectrogram(noise, samplingRate, osc = TRUE)

# Similar, but using the dataframe format to specify a more complicated
# contour for rolloffNoise:
noise = generateNoise(len = samplingRate * 1.2,
  rolloffNoise = data.frame(time = c(0, .3, 1), value = c(-12, 0, -12)),
  noiseFlatSpec = 2000, samplingRate = samplingRate, play = playback)
# spectrogram(noise, samplingRate, osc = TRUE)

# To create a sibilant [s], specify a single strong, broad formant at ~7 kHz:
windowLength_points = 1024
spectralEnvelope = soundgen:::getSpectralEnvelope(
  nr = windowLength_points / 2, nc = 1, samplingRate = samplingRate,
 formants = list('f1' = data.frame(time = 0, freq = 7000,
                                   amp = 50, width = 2000)))
noise = generateNoise(len = samplingRate,
  samplingRate = samplingRate, spectralEnvelope = as.numeric(spectralEnvelope),
  play = playback)
# plot(spectralEnvelope, type = 'l')

# Low-frequency, wind-like noise
spectralEnvelope = soundgen:::getSpectralEnvelope(
  nr = windowLength_points / 2, nc = 1, lipRad = 0,
  samplingRate = samplingRate, formants = list('f1' = data.frame(
    time = 0, freq = 150, amp = 30, width = 90)))
noise = generateNoise(len = samplingRate,
  samplingRate = samplingRate, spectralEnvelope = as.numeric(spectralEnvelope),
  play = playback)

# Manual filter, e.g. for a kettle-like whistle (narrow-band noise)
spectralEnvelope = c(rep(0, 100), 120, rep(0, 100))  # any length is fine
# plot(spectralEnvelope, type = 'b')  # notch filter at Nyquist / 2, here 4 kHz
noise = generateNoise(len = samplingRate, spectralEnvelope = spectralEnvelope,
  samplingRate = samplingRate, play = playback)

# Compare to a similar sound created with soundgen()
# (unvoiced only, a single formant at 4 kHz)
noise_s = soundgen(pitch = NULL,
  noise = data.frame(time = c(0, 1000), value = c(0, 0)),
  formants = list(f1 = data.frame(freq = 4000, amp = 80, width = 20)),
  play = playback)


# Use the spectral envelope of an existing recording (bleating of a sheep)
# (see also the same example with tonal source in ?addFormants)
data(sheep, package = 'seewave')  # import a recording from seewave
sound_orig = as.numeric(sheep@left)
samplingRate = sheep@samp.rate
# playme(sound_orig, samplingRate)

# extract the original spectrogram
windowLength = c(5, 10, 50, 100)[1]  # try both narrow-band (eg 100 ms)
# to get "harmonics" and wide-band (5 ms) to get only formants
spectralEnvelope = spectrogram(sound_orig, windowLength = windowLength,
  samplingRate = samplingRate, output = 'original', padWithSilence = FALSE)
sound_noise = generateNoise(len = length(sound_orig),
  spectralEnvelope = spectralEnvelope, rolloffNoise = 0,
  samplingRate = samplingRate, play = playback)
# playme(sound_noise, samplingRate)

# The spectral envelope is similar to the original recording. Compare:
par(mfrow = c(1, 2))
seewave::meanspec(sound_orig, f = samplingRate, dB = 'max0')
seewave::meanspec(sound_noise, f = samplingRate, dB = 'max0')
par(mfrow = c(1, 1))
# However, the excitation source is now white noise
# (which sounds like noise if windowLength is ~5-10 ms,
# but becomes more and more like the original at longer window lengths)
}
}
\seealso{
\code{\link{soundgen}} \code{\link{fart}} \code{\link{beat}}
}
