#' Generate a Summary Table for ANOVA Results
#'
#' This function creates a summary table for ANOVA results, including degrees of freedom,
#' sum of squares, mean squares, F-values, and p-values. The table is formatted for
#' LaTeX output using the `kableExtra` package.
#'
#' @param model A model object for which ANOVA results are computed (e.g., output from `lm()` or `aov()`).
#' @param caption A character string to be used as the caption for the table.
#'
#' @return A LaTeX-formatted table generated by `kableExtra::kable()`.
#'
#' @importFrom stats anova
#' @importFrom kableExtra kable row_spec column_spec footnote kable_styling
#' @export
#'
#' @examples
#' # Fit a linear model
#' model <- lm(mpg ~ wt + hp, data = mtcars)
#'
#' # Generate the ANOVA summary table
#' ANOVA.summary.table(model, caption = "ANOVA Summary")
ANOVA.summary.table <- function(model, caption) {
  # Compute ANOVA summary
  anova.summary <- anova(model)
  rounding.precision <- 5

  # Create ANOVA table
  anova.table <- data.frame(
    Term = rownames(anova.summary),
    Df = anova.summary$Df,
    Sum.Sq = round(anova.summary$"Sum Sq", rounding.precision),
    Mean.Sq = round(anova.summary$"Mean Sq", rounding.precision),
    F.Value = round(anova.summary$"F value", rounding.precision),
    P.Value = round(anova.summary$"Pr(>F)", rounding.precision)
  )

  # Replace NA values with empty strings
  anova.table[is.na(anova.table)] <- ""

  # Add significance codes
  anova.table$Signif. <- ifelse(anova.table$P.Value < 0.001, ":3",
                                ifelse(anova.table$P.Value < 0.01, ":)",
                                       ifelse(anova.table$P.Value < 0.05, ":/",
                                              ifelse(anova.table$P.Value < 0.1, "", ""))))

  # Format and return LaTeX table
  anova.table[, ] |>
    kableExtra::kable(format = "latex", caption = paste("\\textbf{", caption, "}"), digits = rounding.precision, row.names = FALSE) |>
    kableExtra::row_spec(0, bold = TRUE) |>
    kableExtra::column_spec(1, border_left = TRUE, border_right = FALSE, bold = TRUE) |>
    kableExtra::column_spec(7, border_left = FALSE, border_right = TRUE) |>
    kableExtra::footnote(
      general = c("", "", "", "significance codes -  :3 -  >0.001 ", ":) - >0.01", ":/ - >0.05 ", ""),
      general_title = "",
      footnote_as_chunk = TRUE,
      threeparttable = TRUE
    ) |>
    kableExtra::kable_styling(latex_options = "HOLD_position")
}
