% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ges.R
\name{ges}
\alias{ges}
\title{General Exponential Smoothing}
\usage{
ges(data, orders = c(1, 1), lags = c(1, frequency(data)), type = c("A",
  "M"), persistence = NULL, transition = NULL, measurement = NULL,
  initial = c("optimal", "backcasting"), ic = c("AICc", "AIC", "BIC",
  "BICc"), cfType = c("MSE", "MAE", "HAM", "MSEh", "TMSE", "GTMSE", "MSCE"),
  h = 10, holdout = FALSE, cumulative = FALSE, intervals = c("none",
  "parametric", "semiparametric", "nonparametric"), level = 0.95,
  intermittent = c("none", "auto", "fixed", "interval", "probability", "sba",
  "logistic"), imodel = "MNN", bounds = c("admissible", "none"),
  silent = c("all", "graph", "legend", "output", "none"), xreg = NULL,
  xregDo = c("use", "select"), initialX = NULL, updateX = FALSE,
  persistenceX = NULL, transitionX = NULL, ...)
}
\arguments{
\item{data}{Vector or ts object, containing data needed to be forecasted.}

\item{orders}{Order of the model. Specified as vector of number of states
with different lags. For example, \code{orders=c(1,1)} means that there are
two states: one of the first lag type, the second of the second type.}

\item{lags}{Defines lags for the corresponding orders. If, for example,
\code{orders=c(1,1)} and lags are defined as \code{lags=c(1,12)}, then the
model will have two states: the first will have lag 1 and the second will
have lag 12. The length of \code{lags} must correspond to the length of
\code{orders}.}

\item{type}{Type of model. Can either be \code{"A"} - additive - or
\code{"M"} - multiplicative. The latter means that the GES is fitted on
log-transformed data.}

\item{persistence}{Persistence vector \eqn{g}, containing smoothing
parameters. If \code{NULL}, then estimated.}

\item{transition}{Transition matrix \eqn{F}. Can be provided as a vector.
Matrix will be formed using the default \code{matrix(transition,nc,nc)},
where \code{nc} is the number of components in state vector. If \code{NULL},
then estimated.}

\item{measurement}{Measurement vector \eqn{w}. If \code{NULL}, then
estimated.}

\item{initial}{Can be either character or a vector of initial states. If it
is character, then it can be \code{"optimal"}, meaning that the initial
states are optimised, or \code{"backcasting"}, meaning that the initials are
produced using backcasting procedure.}

\item{ic}{The information criterion used in the model selection procedure.}

\item{cfType}{Type of Cost Function used in optimization. \code{cfType} can
be: \code{MSE} (Mean Squared Error), \code{MAE} (Mean Absolute Error),
\code{HAM} (Half Absolute Moment), \code{TMSE} - Trace Mean Squared Error,
\code{GTMSE} - Geometric Trace Mean Squared Error, \code{MSEh} - optimisation
using only h-steps ahead error, \code{MSCE} - Mean Squared Cumulative Error.
If \code{cfType!="MSE"}, then likelihood and model selection is done based
on equivalent \code{MSE}. Model selection in this cases becomes not optimal.

There are also available analytical approximations for multistep functions:
\code{aMSEh}, \code{aTMSE} and \code{aGTMSE}. These can be useful in cases
of small samples.

Finally, just for fun the absolute and half analogues of multistep estimators
are available: \code{MAEh}, \code{TMAE}, \code{GTMAE}, \code{MACE}, \code{TMAE},
\code{HAMh}, \code{THAM}, \code{GTHAM}, \code{CHAM}.}

\item{h}{Length of forecasting horizon.}

\item{holdout}{If \code{TRUE}, holdout sample of size \code{h} is taken from
the end of the data.}

\item{cumulative}{If \code{TRUE}, then the cumulative forecast and prediction
intervals are produced instead of the normal ones. This is useful for
inventory control systems.}

\item{intervals}{Type of intervals to construct. This can be:

\itemize{
\item \code{none}, aka \code{n} - do not produce prediction
intervals.
\item \code{parametric}, \code{p} - use state-space structure of ETS. In
case of mixed models this is done using simulations, which may take longer
time than for the pure additive and pure multiplicative models.
\item \code{semiparametric}, \code{sp} - intervals based on covariance
matrix of 1 to h steps ahead errors and assumption of normal / log-normal
distribution (depending on error type).
\item \code{nonparametric}, \code{np} - intervals based on values from a
quantile regression on error matrix (see Taylor and Bunn, 1999). The model
used in this process is e[j] = a j^b, where j=1,..,h.
}
The parameter also accepts \code{TRUE} and \code{FALSE}. The former means that
parametric intervals are constructed, while the latter is equivalent to
\code{none}.
If the forecasts of the models were combined, then the intervals are combined
quantile-wise (Lichtendahl et al., 2013).}

\item{level}{Confidence level. Defines width of prediction interval.}

\item{intermittent}{Defines type of intermittent model used. Can be: 1.
\code{none}, meaning that the data should be considered as non-intermittent;
2. \code{fixed}, taking into account constant Bernoulli distribution of
demand occurrences; 3. \code{interval}, Interval-based model, underlying
Croston, 1972 method; 4. \code{probability}, Probability-based model,
underlying Teunter et al., 2011 method. 5. \code{auto} - automatic selection
of intermittency type based on information criteria. The first letter can be
used instead. 6. \code{"sba"} - Syntetos-Boylan Approximation for Croston's
method (bias correction) discussed in Syntetos and Boylan, 2005. 7.
\code{"logistic"} - the probability is estimated based on logistic regression
model principles.}

\item{imodel}{Type of ETS model used for the modelling of the time varying
probability. Object of the class "iss" can be provided here, and its parameters
would be used in iETS model.}

\item{bounds}{What type of bounds to use in the model estimation. The first
letter can be used instead of the whole word.}

\item{silent}{If \code{silent="none"}, then nothing is silent, everything is
printed out and drawn. \code{silent="all"} means that nothing is produced or
drawn (except for warnings). In case of \code{silent="graph"}, no graph is
produced. If \code{silent="legend"}, then legend of the graph is skipped.
And finally \code{silent="output"} means that nothing is printed out in the
console, but the graph is produced. \code{silent} also accepts \code{TRUE}
and \code{FALSE}. In this case \code{silent=TRUE} is equivalent to
\code{silent="all"}, while \code{silent=FALSE} is equivalent to
\code{silent="none"}. The parameter also accepts first letter of words ("n",
"a", "g", "l", "o").}

\item{xreg}{Vector (either numeric or time series) or matrix (or data.frame)
of exogenous variables that should be included in the model. If matrix
included than columns should contain variables and rows - observations. Note
that \code{xreg} should have number of observations equal either to
in-sample or to the whole series. If the number of observations in
\code{xreg} is equal to in-sample, then values for the holdout sample are
produced using \link[smooth]{es} function.}

\item{xregDo}{Variable defines what to do with the provided xreg:
\code{"use"} means that all of the data should be used, while
\code{"select"} means that a selection using \code{ic} should be done.
\code{"combine"} will be available at some point in future...}

\item{initialX}{Vector of initial parameters for exogenous variables.
Ignored if \code{xreg} is NULL.}

\item{updateX}{If \code{TRUE}, transition matrix for exogenous variables is
estimated, introducing non-linear interactions between parameters.
Prerequisite - non-NULL \code{xreg}.}

\item{persistenceX}{Persistence vector \eqn{g_X}, containing smoothing
parameters for exogenous variables. If \code{NULL}, then estimated.
Prerequisite - non-NULL \code{xreg}.}

\item{transitionX}{Transition matrix \eqn{F_x} for exogenous variables. Can
be provided as a vector. Matrix will be formed using the default
\code{matrix(transition,nc,nc)}, where \code{nc} is number of components in
state vector. If \code{NULL}, then estimated. Prerequisite - non-NULL
\code{xreg}.}

\item{...}{Other non-documented parameters.  For example parameter
\code{model} can accept a previously estimated GES model and use all its
parameters.  \code{FI=TRUE} will make the function produce Fisher
Information matrix, which then can be used to calculated variances of
parameters of the model.
You can also pass two parameters to the optimiser: 1. \code{maxeval} - maximum
number of evaluations to carry on; 2. \code{xtol_rel} - the precision of the
optimiser. The default values used in es() are \code{maxeval=5000} and
\code{xtol_rel=1e-8}. You can read more about these parameters in the
documentation of \link[nloptr]{nloptr} function.}
}
\value{
Object of class "smooth" is returned. It contains:

\itemize{
\item \code{model} - name of the estimated model.
\item \code{timeElapsed} - time elapsed for the construction of the model.
\item \code{states} - matrix of fuzzy components of GES, where \code{rows}
correspond to time and \code{cols} to states.
\item \code{initialType} - Type of the initial values used.
\item \code{initial} - initial values of state vector (extracted from
\code{states}).
\item \code{nParam} - table with the number of estimated / provided parameters.
If a previous model was reused, then its initials are reused and the number of
provided parameters will take this into account.
\item \code{measurement} - matrix w.
\item \code{transition} - matrix F.
\item \code{persistence} - persistence vector. This is the place, where
smoothing parameters live.
\item \code{fitted} - fitted values of ETS.
\item \code{forecast} - point forecast of ETS.
\item \code{lower} - lower bound of prediction interval. When
\code{intervals="none"} then NA is returned.
\item \code{upper} - higher bound of prediction interval. When
\code{intervals="none"} then NA is returned.
\item \code{residuals} - the residuals of the estimated model.
\item \code{errors} - matrix of 1 to h steps ahead errors.
\item \code{s2} - variance of the residuals (taking degrees of freedom
into account).
\item \code{intervals} - type of intervals asked by user.
\item \code{level} - confidence level for intervals.
\item \code{cumulative} - whether the produced forecast was cumulative or not.
\item \code{actuals} - original data.
\item \code{holdout} - holdout part of the original data.
\item \code{imodel} - model of the class "iss" if intermittent model was estimated.
If the model is non-intermittent, then imodel is \code{NULL}.
\item \code{xreg} - provided vector or matrix of exogenous variables. If
\code{xregDo="s"}, then this value will contain only selected exogenous variables.
\item \code{updateX} - boolean, defining, if the states of exogenous variables
were estimated as well.
\item \code{initialX} - initial values for parameters of exogenous variables.
\item \code{persistenceX} - persistence vector g for exogenous variables.
\item \code{transitionX} - transition matrix F for exogenous variables.
\item \code{ICs} - values of information criteria of the model. Includes
AIC, AICc, BIC and BICc.
\item \code{logLik} - log-likelihood of the function.
\item \code{cf} - Cost function value.
\item \code{cfType} - Type of cost function used in the estimation.
\item \code{FI} - Fisher Information. Equal to NULL if \code{FI=FALSE} or
when \code{FI} variable is not provided at all.
\item \code{accuracy} - vector of accuracy measures for the holdout sample.
In case of non-intermittent data includes: MPE, MAPE, SMAPE, MASE, sMAE,
RelMAE, sMSE and Bias coefficient (based on complex numbers). In case of
intermittent data the set of errors will be: sMSE, sPIS, sCE (scaled
cumulative error) and Bias coefficient. This is available only when
\code{holdout=TRUE}.
}
}
\description{
Function constructs General Exponential Smoothing, estimating matrices F, w,
vector g and initial parameters.
}
\details{
The function estimates the Single Source of Error state space model of the
following type:

\deqn{y_{t} = o_{t} (w' v_{t-l} + x_t a_{t-1} + \epsilon_{t})}

\deqn{v_{t} = F v_{t-l} + g \epsilon_{t}}

\deqn{a_{t} = F_{X} a_{t-1} + g_{X} \epsilon_{t} / x_{t}}

Where \eqn{o_{t}} is the Bernoulli distributed random variable (in case of
normal data equal to 1), \eqn{v_{t}} is the state vector (defined using
\code{orders}) and \eqn{l} is the vector of \code{lags}, \eqn{x_t} is the
vector of exogenous parameters. \eqn{w} is the \code{measurement} vector,
\eqn{F} is the \code{transition} matrix, \eqn{g} is the \code{persistence}
vector, \eqn{a_t} is the vector of parameters for exogenous variables,
\eqn{F_{X}} is the \code{transitionX} matrix and \eqn{g_{X}} is the
\code{persistenceX} matrix. Finally, \eqn{\epsilon_{t}} is the error term.
}
\examples{

# Something simple:
ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,bounds="a",intervals="p")

# A more complicated model with seasonality
\dontrun{ourModel <- ges(rnorm(118,100,3),orders=c(2,1),lags=c(1,4),h=18,holdout=TRUE)}

# Redo previous model on a new data and produce prediction intervals
\dontrun{ges(rnorm(118,100,3),model=ourModel,h=18,intervals="sp")}

# Produce something crazy with optimal initials (not recommended)
\dontrun{ges(rnorm(118,100,3),orders=c(1,1,1),lags=c(1,3,5),h=18,holdout=TRUE,initial="o")}

# Simpler model estiamted using trace forecast error cost function and its analytical analogue
\dontrun{ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,bounds="n",cfType="TMSE")
ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,bounds="n",cfType="aTMSE")}

# Introduce exogenous variables
\dontrun{ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,xreg=c(1:118))}

# Ask for their update
\dontrun{ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,xreg=c(1:118),updateX=TRUE)}

# Do the same but now let's shrink parameters...
\dontrun{ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,xreg=c(1:118),updateX=TRUE,cfType="TMSE")
ourModel <- ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,cfType="aTMSE")}

# Or select the most appropriate one
\dontrun{ges(rnorm(118,100,3),orders=c(1),lags=c(1),h=18,holdout=TRUE,xreg=c(1:118),xregDo="s")

summary(ourModel)
forecast(ourModel)
plot(forecast(ourModel))}

}
\references{
\itemize{
\item Svetunkov I. (2017). Statistical models underlying functions of 'smooth'
package for R. Working Paper of Department of Management Science, Lancaster
University 2017:1, 1-52.
}

\itemize{
\item Taylor, J.W. and Bunn, D.W. (1999) A Quantile Regression Approach to
Generating Prediction Intervals. Management Science, Vol 45, No 2, pp
225-237.
\item Lichtendahl Kenneth C., Jr., Grushka-Cockayne Yael, Winkler
Robert L., (2013) Is It Better to Average Probabilities or
Quantiles? Management Science 59(7):1594-1611. DOI:
[10.1287/mnsc.1120.1667](https://doi.org/10.1287/mnsc.1120.1667)
}
}
\seealso{
\code{\link[forecast]{ets}, \link[smooth]{es}, \link[smooth]{ces},
\link[smooth]{sim.es}}
}
\author{
Ivan Svetunkov, \email{ivan@svetunkov.ru}
}
\keyword{models}
\keyword{nonlinear}
\keyword{regression}
\keyword{smooth}
\keyword{ts}
\keyword{univar}
